/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';


import ServerSideRender from '@wordpress/server-side-render';
import { Disabled } from '@wordpress/components';
import metadata from './block.json';
import { InspectorControls, RichText, useBlockProps } from "@wordpress/block-editor";
import { PanelBody, ColorPicker, __experimentalBoxControl as BoxControl } from '@wordpress/components';



/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function edit({attributes, setAttributes}) {
  const { title, description, jobslimit, bgColor, padding } = attributes;
 
  const blockProps = useBlockProps({
      style: {
          backgroundColor: bgColor,
          padding: `${padding?.top || 0}px ${padding?.right || 0}px ${padding?.bottom || 0}px ${padding?.left || 0}px`,
      },
  });
 
	return (
        <div { ...blockProps }>
        <InspectorControls>
            <PanelBody
                title={__('Jobs Limit', 'ai-job-listing')}
                initialOpen={true}
            >
                <RichText
                    className="wp-block-wp-jobs_jobslimit"
                    tagName="div"
                    placeholder={__("Jobs Limits", "ai-job-listing")}
                    value={jobslimit}
                    onChange={(jobslimit) => setAttributes({ jobslimit })}
                />
            </PanelBody>

            <PanelBody
                title={__('Color Settings', 'ai-job-listing')}
                initialOpen={true}
            >
                <ColorPicker
                    color={bgColor}
                    onChange={(bgColor) => setAttributes({ bgColor })}
                    enableAlpha
                    defaultValue={bgColor}
                    clearable={false}
                />
            </PanelBody>

            <PanelBody
                title={__('Padding/Margin Settings', 'ai-job-listing')}
                initialOpen={false}
            >
                <BoxControl
                    label={__('Inline Padding', 'ai-job-listing')}
                    values={padding}
                    onChange={(newPadding) => setAttributes({ padding: newPadding })}
                />
            </PanelBody>
        </InspectorControls>

        <Disabled>
            <ServerSideRender
                block={metadata.name}
                skipBlockSupportAttributes
                attributes={attributes}
            />
        </Disabled>

        <RichText
            tagName="h2"
            className="wp-block-wp-jobs_title"
            placeholder={__('Enter title...', 'ai-job-listing')}
            value={title}
            onChange={(title) => setAttributes({ title })}
        />
        <RichText
            tagName="p"
            className="wp-block-wp-jobs_description"
            placeholder={__('Enter description...', 'ai-job-listing')}
            value={description}
            onChange={(description) => setAttributes({ description })}
        />
    </div>
	);
}
