/**
 * External dependencies.
 */
import { twMerge } from 'tailwind-merge';
import { Link } from 'react-router-dom';

/**
 * Internal dependencies.
 */
import { TButton } from '../../interfaces';

const Button = (props: TButton) => {
    const {
        text = '',
        href = undefined,
        externalLink = false,
        icon = '',
        type = 'button',
        btnStyle = 'default',
        outline = false,
        buttonCustomClass = '',
        disabled = false,
        onClick = () => {},
        size = 'default',
    } = props;

    const getClassName = () => {
        let className = `inline-flex items-center whitespace-nowrap gap-1 transition decoration-transparent leading-[1]`;
        let sizeClass = ``;
        switch (size) {
            case 'large':
                sizeClass = `px-[25px] font-bold text-[15px] min-h-[47px]`;
                break;
            case 'mid':
                sizeClass = `px-[25px] font-bold text-[14px] min-h-[42px]`;
                break;
            case 'default':
                sizeClass = `px-5 py-2.5 font-bold text-[13px] min-h-[38px]`;
                break;
            default:
                break;
        }
        if (btnStyle === 'primary') {
            className += ` rounded-[100px] !text-white border bg-primary border-primary ${sizeClass}`;
            className += ` hover:text-white hover:bg-primary`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'secondary') {
            className += ` rounded-[100px] !text-white border bg-secondary border-secondary ${sizeClass}`;
            className += ` hover:text-white hover:bg-secondary`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'secondary' && outline) {
            className += ` border border-secondary !text-secondary bg-transparent ${sizeClass}`;
            className += ` hover:!text-white`;
        }
        if (btnStyle === 'light' && outline) {
            className += ` rounded-[100px] text-b2-table border border-border-mid ${sizeClass}`;
            className += ` hover:text-white hover:bg-secondary hover:border-secondary`;
            className += ` focus:outline-none focus:shadow-none`;
        }

        if (btnStyle === 'action' && outline) {
            className += ` px-2.5 py-[7px] rounded-xsm text-[15px] text-primary border border-border-mid`;
            className += ` hover:bg-primary hover:border-primary hover:text-white`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'actionSecondary' && outline) {
            className += ` px-2.5 py-[7px] rounded-xsm text-[15px] text-secondary border border-border-mid`;
            className += ` hover:bg-secondary hover:border-secondary hover:text-white`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'transparentDanger') {
            className += ` px-2.5 py-[5.5px] rounded-xsm text-[15px] text-danger`;
            className += ` hover:text-danger`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'transparentSecondary') {
            className += ` p-[6px] text-[15px] text-secondary`;
            className += ` hover:!underline hover:!underline-offset-2 hover:!decoration-secondary`;
            className += ` focus:outline-none focus:shadow-none`;
        }
        if (btnStyle === 'transparentPrimary') {
            className += ` p-[6px] text-[15px] text-primary`;
            className += ` hover:!underline hover:!underline-offset-2 hover:!decoration-primary`;
            className += ` focus:outline-none focus:shadow-none`;
        }

        // Add opacity for disabled button
        if (disabled) {
            className += ' opacity-50 cursor-not-allowed pointer-events-none';
        }

        // Add custom class name if provided
        if (
            typeof buttonCustomClass !== 'undefined' &&
            buttonCustomClass !== null
        ) {
            className = `${buttonCustomClass} ${className}`;
        }

        return twMerge(className);
    };

    const generateButton = () => {
        if (typeof href !== 'undefined' && !externalLink) {
            return (
                <Link to={href} className={getClassName()} onClick={onClick}>
                    {icon && <i className={icon}></i>}
                    {text}
                </Link>
            );
        } else if (externalLink) {
            return (
                <a
                    href={href}
                    target="_blank"
                    className={getClassName()}
                    onClick={onClick}
                    rel="noopener noreferrer"
                >
                    {icon && <i className={icon}></i>}
                    {text}
                </a>
            );
        } else if (typeof href === 'undefined' && !externalLink) {
            return (
                <button
                    type={type}
                    onClick={onClick}
                    className={getClassName()}
                >
                    {icon && <i className={icon}></i>}
                    {text}
                </button>
            );
        }
    };

    return <>{generateButton()}</>;
};

export default Button;
