/**
 * External dependencies
 */
import { useState, useRef, useEffect } from '@wordpress/element';
import { Calendar } from 'react-date-range';
import { format } from 'date-fns';

/**
 * Internal dependencies
 */
// Import the stylesheet for the plugin.
import 'react-date-range/dist/styles.css'; // main css file
import 'react-date-range/dist/theme/default.css'; // theme css file

interface type {
    onChange?: (val: any) => void;
    id: string;
    minDate?: Date | undefined;
    defaultValue?: string | number | Date;
}

const DatePicker = (props: type) => {
    const {
        onChange = () => {},
        id = '',
        minDate = undefined,
        defaultValue = '',
    } = props;
    const ref = useRef(null);
    const [showDate, setShowDate] = useState(false);
    const [calendarDate, setCalendarDate] = useState<any>(null);

    const handleClickOutside = (event: React.ChangeEvent<HTMLInputElement>) => {
        if (ref.current && !ref.current.contains(event.target)) {
            setShowDate(false);
        }
    };

    const getCurrentTime = () => {
        const currentDate = new Date();
        const hours = currentDate.getHours();
        const minute = currentDate.getMinutes();
        const second = currentDate.getSeconds();
        return `${hours}:${minute}:${second}`;
    };

    const handleDatePicker = (date: any) => {
        const dateFormated = format(date, 'yyyy-MM-dd');
        const currentTime = getCurrentTime();
        const finalDate = {
            dateOnly: dateFormated,
            timeOnly: currentTime,
        };
        setCalendarDate(finalDate.dateOnly);
        onChange({
            name: id,
            value: {
                ...finalDate,
                value: finalDate.dateOnly + ' ' + finalDate.timeOnly,
            },
        });
        setShowDate(!showDate);
    };

    /**
     * Handle Outside click of the component.
     */
    useEffect(() => {
        document.addEventListener('click', handleClickOutside, true);
        return () => {
            document.removeEventListener('click', handleClickOutside, true);
        };
    });

    useEffect(() => {
        if (defaultValue) {
            const stringToDate = new Date(defaultValue);
            setCalendarDate(
                !isNaN(stringToDate) ? format(stringToDate, 'yyyy-MM-dd') : ''
            );
        }
    }, [defaultValue]);

    return (
        <>
            <div ref={ref} className="relative" id={id}>
                <button
                    type="button"
                    onClick={() => setShowDate(!showDate)}
                    className="w-full h-[42px] flex items-center justify-between px-5 py-1 border border-border-dark rounded cursor-pointer"
                >
                    <span className="text-[14px] text-headline">
                        {calendarDate ?? <div className="text-b2-table"></div>}
                    </span>
                    <span className="text-[16px] text-b1-notice-comment">
                        <i className="tio-calendar_month"></i>
                    </span>
                </button>
                {showDate && (
                    <Calendar
                        date={
                            defaultValue ? new Date(defaultValue) : new Date()
                        }
                        minDate={minDate}
                        onChange={handleDatePicker}
                    />
                )}
            </div>
        </>
    );
};

export default DatePicker;
