/**
 * External dependencies.
 */
import { twMerge } from 'tailwind-merge';

/**
 * Internal dependencies
 */
import Label from './InputLabel';
import Select2Input, { Select2SingleRow } from './Select2Input';
import TextEditor from './TextEditor';

export type IInputResponse = {
    /**
     * Input ID or name.
     */
    name: string;

    /**
     * Value after onChange input.
     */
    value: any;
};

type InputType =
    | 'text'
    | 'url'
    | 'email'
    | 'radio'
    | 'search'
    | 'password'
    | 'number'
    | 'tel'
    | 'color'
    | 'file'
    | 'select'
    | 'textarea'
    | 'switch'
    | 'multiselect'
    | 'checkbox'
    | 'text-editor'
    | undefined;

export interface InputProps {
    /**
     * Input type - text, email, password, number, textarea, select, switch, multiselect, currency
     */
    type?: InputType;

    /**
     * Input label - If given, then a <Label component will be rendered at top
     */
    label?: string;

    /**
     * Input label class if given label.
     */
    labelClass?: string;

    /**
     * Input name or ID.
     */
    id?: string;

    /**
     * Input name or ID.
     */
    name?: string;

    /**
     * Input shown value.
     */
    value?: string | number | Array<any> | boolean | null;

    /**
     * Input placeholder text.
     */
    placeholder?: React.CSSProperties['placeholder'];

    /**
     * Is input required or not.
     */
    required?: boolean;

    /**
     * Input class.
     */
    className?: React.CSSProperties['className'];

    /**
     * Input Area class.
     */
    areaClassName?: React.CSSProperties['className'];

    /**
     * Ref
     */
    ref?: LegacyRef<HTMLInputElement> | null;

    /**
     * If select2 is type, then pass options.
     */
    options?: Array<Select2SingleRow>;

    /**
     * On change input.
     */
    onChange?: (val: any) => void;

    /**
     * On input handle.
     */
    onInput?: (val: any) => void;
    /**
     * On input handle.
     */
    onKeyup?: (val: any) => void;

    /**
     * For textarea input, rows.
     */
    rows?: number;

    /**
     * For textarea input, cols.
     */
    cols?: number;

    /**
     * Custom button style.
     */
    style?: any;

    /**
     * Autocomplete enable or not.
     */
    autoComplete?: string;

    /**
     * Min value.
     */
    min?: number;

    /**
     * Max value.
     */
    max?: number;

    /**
     * Left text or node.
     */
    left?: string | React.ReactNode;

    /**
     * Right text or node.
     */
    right?: string | React.ReactNode;

    /**
     * Height of editor-textarea.
     */
    editorHeight?: string;

    /**
     * Show error message or not.
     */
    hasError?: boolean;

    /**
     * Error message.
     */
    errorMessage?: string;

    /**
     * Checked.
     */
    checked?: boolean | undefined;

    /**
     * Checked.
     */
    readOnly?: boolean | undefined;

    /**
     * Select 2 Style.
     */
    select2Style?: any;
    /**
     * Select 2 Searchable.
     */
    isSearchable?: boolean;
}

export function Input(props: InputProps) {
    const {
        type = 'text',
        label = '',
        labelClass = '',
        placeholder = '',
        className = '',
        areaClassName = '',
        onChange = () => {},
        onInput = () => {},
        onKeyup = () => {},
        id = '',
        name = '',
        value = '',
        required = false,
        ref = null,
        options = [],
        rows = 4,
        cols = 90,
        style = {},
        autoComplete = 'off',
        min = 0,
        max,
        left = <></>,
        right = <></>,
        editorHeight = '150px',
        hasError = false,
        errorMessage = '',
        checked = undefined,
        select2Style = undefined,
        readOnly = undefined,
        isSearchable = false,
    } = props;

    const getClassName = () => {
        let classNames = ``;

        switch (type) {
            case 'text':
            case 'url':
            case 'number':
            case 'email':
            case 'search':
            case 'password':
            case 'tel':
            case 'color':
            case 'file':
                classNames += `!min-h-[42px] bg-white placeholder:text-b2-table transition !py-1 !px-5 !rounded
                                text-b3-regular !text-b1-notice-comment !border !border-solid !border-border-dark !shadow-none placeholder:text-b2-table focus:!outline-none 
                                focus:!border-primary focus:!shadow-none focus:ring-0 required:border-red-500 w-full shadow-none !border-1`;
                break;

            case 'textarea':
                classNames += `w-full border-border-mid rounded`;
                break;
            case 'radio':
                classNames += `!m-0 border-2 !border-border-dark text-primary focus:ring-0 focus:![box-shadow:none] focus:text-primary checked:text-primary checked:!border-primary checked:before:!bg-white`;
                break;

            case 'checkbox':
                classNames += `!h-4 !w-4 !mr-2.5 !border-2 !border-border-dark !rounded-[2px] !text-primary cursor-pointer focus:![box-shadow:none] checked:bg-checkbox-size checked:bg-checkbox-position checked:bg-checkbox-check checked:before:!content-none checked:!border-primary`;
                break;

            default:
                break;
        }

        if (className?.length) {
            classNames = `${classNames} ${className}`;
        }

        return twMerge(classNames);
    };
    const updatedStyle = {
        ...style,
    };

    const changeInput = (e: any) => {
        let { name, value: targetValue }: IInputResponse = e.target;

        if ('checkbox' === type) {
            targetValue = e.target.checked ? true : false;
        }

        onChange({ name, value: targetValue });
    };

    return (
        <>
            {label.length ? (
                <Label htmlFor={id} className={labelClass}>
                    {label} {required ? <span>*</span> : ''}
                </Label>
            ) : (
                ''
            )}

            <div className={`flex items-center relative ${areaClassName}`}>
                {left}
                {(type === 'text' ||
                    type === 'url' ||
                    type === 'number' ||
                    type === 'radio' ||
                    type === 'checkbox' ||
                    type === 'email' ||
                    type === 'password' ||
                    type === 'tel' ||
                    type === 'file' ||
                    type === 'color' ||
                    type === 'search') && (
                    <input
                        type={type}
                        ref={ref}
                        required={required}
                        placeholder={placeholder}
                        className={getClassName()}
                        style={updatedStyle}
                        onChange={changeInput}
                        onInput={onInput}
                        name={name}
                        id={id}
                        value={value}
                        min={min}
                        max={max}
                        autoComplete={autoComplete}
                        checked={checked}
                        readOnly={readOnly}
                    />
                )}

                {type === 'text-editor' && (
                    <div className="w-full">
                        <TextEditor
                            id={id}
                            height={editorHeight}
                            onChange={onChange}
                            value={value}
                            placeholder={placeholder}
                        />
                    </div>
                )}

                {type === 'textarea' && (
                    <textarea
                        ref={ref}
                        required={required}
                        placeholder={placeholder}
                        className={getClassName()}
                        style={updatedStyle}
                        onChange={changeInput}
                        onInput={onInput}
                        onKeyUp={onKeyup}
                        id={id}
                        name={name}
                        value={value}
                        rows={rows}
                        cols={cols}
                        autoComplete={autoComplete}
                    ></textarea>
                )}

                {(type === 'select' || type === 'multiselect') && (
                    <Select2Input
                        defaultValue={value}
                        isMulti={type === 'multiselect'}
                        options={typeof options !== 'undefined' ? options : []}
                        placeholder={placeholder}
                        style={select2Style}
                        isSearchable={isSearchable}
                        onChange={(val) => {
                            if (typeof onChange === 'function') {
                                onChange({ name, value: val });
                            }
                        }}
                    />
                )}
                {right}
            </div>
            {errorMessage && (
                <div className="text-error text-b5-regular mt-1.5">
                    {errorMessage}
                </div>
            )}
        </>
    );
}
