/**
 * External dependencies
 */
import Select from 'react-select';
import makeAnimated from 'react-select/animated';
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies
 */
import {
    formatSelect2Data,
    getSelectedOption,
} from '../../utils/Select2Helper';
import { selectStyles } from './StyleClass';

export interface Select2SingleRow {
    /**
     * Select2 option label.
     */
    label: string;

    /**
     * Select2 option value.
     */
    value: number | string | boolean;
}

export interface ISelect2Input {
    /**
     * Select2 options.
     */
    options: Array<Select2SingleRow>;

    /**
     * Placeholder text.
     */
    placeholder?: string;

    /**
     * Is Multi-Select or not.
     */
    isMulti?: boolean;

    /**
     * Is Searchable or not.
     */
    isSearchable?: boolean;

    /**
     * Default selected value.
     */
    defaultValue?: any;

    /**
     * On change select2 input.
     */
    onChange?: (val: any) => void;

    /**
     * On change select2 input.
     */
    style?: any;
}

const Select2Input = (props: ISelect2Input) => {
    const {
        options,
        isMulti,
        isSearchable,
        placeholder,
        defaultValue,
        onChange,
        style,
    } = props;

    const animatedComponents = makeAnimated();

    return (
        <Select
            styles={style ?? selectStyles}
            components={animatedComponents}
            options={formatSelect2Data(options)}
            isMulti={isMulti}
            isSearchable={isSearchable ?? false}
            value={
                isMulti
                    ? formatSelect2Data(defaultValue)
                    : getSelectedOption(options, defaultValue)
            }
            defaultValue={
                isMulti
                    ? formatSelect2Data(defaultValue)
                    : getSelectedOption(options, defaultValue)
            }
            placeholder={placeholder ? placeholder : __('-', 'ai-job-listing')}
            onChange={(value) => {
                if (typeof onChange === 'function') {
                    onChange(value);
                }
            }}
        />
    );
};

Select2Input.defaultProps = {
    options: [],
    placeholder: '',
    isMulti: false,
};

export default Select2Input;
