/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { Switch } from '@headlessui/react';
import { __ } from '@wordpress/i18n';

interface switchProps {
    initialValue: boolean;
    id: string;
    onChange?: (event: { name: string; value: boolean }) => void;
    sizeLg?: boolean;
    parentClass?: string;
    label?: boolean;
}

export default function SwitchCheckbox(props: switchProps) {
    const {
        initialValue = false,
        onChange = () => {},
        id = '',
        sizeLg = false,
        parentClass = '',
        label = false,
    } = props;

    const [enabled, setEnabled] = useState(initialValue);
    useEffect(() => {
        setEnabled(initialValue);
    }, [initialValue]);

    const handleToggleChange = (value: any) => {
        setEnabled(value);
        if (typeof onChange === 'function') {
            onChange({ name: id, value });
        }
    };

    let checkedStyle = '';

    if (enabled && !sizeLg) {
        checkedStyle = 'translate-x-[12px] bg-white';
    } else if (!enabled && !sizeLg) {
        checkedStyle = 'translate-x-0 bg-white';
    } else if (enabled && sizeLg) {
        checkedStyle = 'translate-x-[20px] bg-white';
    } else if (!enabled && sizeLg) {
        checkedStyle = 'translate-x-[3px] bg-white';
    }

    return (
        <div className={parentClass}>
            <Switch
                checked={enabled}
                onChange={handleToggleChange}
                className={`${
                    enabled
                        ? '!bg-primary border-primary'
                        : '!bg-border-mid border-border-mid'
                } ${sizeLg ? 'h-[22px] w-[40px]' : 'h-[14px] w-[26px]'}
                relative inline-flex flex-shrink-0 border-2 rounded-[100px] cursor-pointer transition-colors ease-in-out duration-200 focus:outline-none focus-visible:ring-2  focus-visible:ring-white focus-visible:ring-opacity-75`}
            >
                <span className="sr-only">
                    {__('Toggle', 'ai-job-listing')}
                </span>
                <span
                    aria-hidden="true"
                    className={`${checkedStyle} ${
                        sizeLg
                            ? 'h-[14px] w-[14px] top-[2px]'
                            : 'h-[10px] w-[10px] top-0'
                    } pointer-events-none inline-block rounded-full transform ring-0 transition ease-in-out duration-200 mr-[2px] absolute left-0`}
                />
            </Switch>
            {label && (
                <span
                    className={`text-[14px] ml-[5px] ${
                        enabled ? 'text-b2-table' : 'text-b2-table'
                    }`}
                >
                    {enabled ? 'Active' : 'Inactive'}
                </span>
            )}
        </div>
    );
}
