/**
 * External dependencies
 */
import { useSelect } from '@wordpress/data';
import { useState, useEffect } from '@wordpress/element';
const nl2br = require('react-nl2br');

/**
 * Internal dependencies.
 */
import Modal from './Modal';
import jobstore from '../../data/job';
import LoadingBox from '../loader/Loading';
import { formatSalaryAmount, getApplicantValueByKey, getFormattedUrl } from '../../utils/Global';
import generalSettings from '../../data/general-settings';
import { formatDateOnly } from '../../utils/DateHelper';
import { Input } from '../form/Input';
import PdfViewer from './PdfViewer';

type Props = {
    isPreviewModalOpen: boolean;
    closePreviewModal: () => void;
    jobSingleApplication: object;
    submitCommentHandler: (val: any) => void;
    commentText: string;
    commentFormError: string;
    setCommentText: (val: any) => void;
    applicationComments: Array<any>;
};

const ApplicationPreviewModal = ({
    isPreviewModalOpen,
    closePreviewModal,
    jobSingleApplication,
    commentText = '',
    commentFormError = '',
    setCommentText = () => {},
    submitCommentHandler = () => {},
    applicationComments = [],
}: Props) => {
    const jobApplicationDetailsLoading: boolean = useSelect(
        (select) => select(jobstore).getJobApplicationDetailsLoading(),
        []
    );
    const getUser: object = useSelect(
        (select) => select(generalSettings).getCurrentUser(),
        []
    );
    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    // Resume Preview Modal 
    const [isPdfViewModalOpen, setIsPdfViewModalOpen] = useState(false);
    const [isModalOpen, setIsModalOpen] = useState(false);

    useEffect(() => {
        setIsModalOpen(isPreviewModalOpen);
    }, [isPreviewModalOpen]);

    const pdfViewerModalHandler = () => {
        setIsPdfViewModalOpen(true);
        setIsModalOpen(false);
    }
    const closePdfViewModal = () => {
        setIsPdfViewModalOpen(false);
        setIsModalOpen(true);
    }

    return (
        <>
            <Modal
                isOpen={isModalOpen}
                closeModal={closePreviewModal}
                areaClass="max-w-[1400px] !p-0 mt-[50px]"
            >
                {jobApplicationDetailsLoading ? (
                    <LoadingBox block={10} />
                ) : (
                    <>
                        <div className="relative p-[50px] bg-background break-all">
                            <button
                                onClick={closePreviewModal}
                                className="absolute flex items-center justify-center top-[10px] right-[10px] h-8 w-8 rounded-full bg-primary text-white"
                            >
                                <i className="tio-clear"></i>
                            </button>
                            <div className=" h-[75vh] overflow-y-auto">
                                <div className="flex gap-5">
                                    <div className="flex-1">
                                        {/* Applicant Contact Info */}
                                        <div className="mb-5 bg-white rounded p-[30px]">
                                            <div className="flex items-center gap-[30px]">
                                                {jobSingleApplication.profile_image_url ? (
                                                    <img
                                                        className="object-cover rounded-full h-[110px] w-[110px] m-0"
                                                        src={
                                                            jobSingleApplication.profile_image_url
                                                        }
                                                        alt={getApplicantValueByKey(
                                                            jobSingleApplication?.meta_attributes,
                                                            'full_name'
                                                        )}
                                                    />
                                                ) : (
                                                    <span className="rounded-full h-[110px] leading-[1]">
                                                        <i className="tio-account_circle text-[#c5c5c5] text-[110px]"></i>
                                                    </span>
                                                )}
                                                <div className="flex-1">
                                                    <div className="mb-[5px] text-[24px] leading-[34px] font-bold text-headline">
                                                        {getApplicantValueByKey(
                                                            jobSingleApplication?.meta_attributes,
                                                            'full_name'
                                                        )}
                                                    </div>
                                                    {getApplicantValueByKey(
                                                        jobSingleApplication?.meta_attributes,
                                                        'email'
                                                    ) && (
                                                        <a
                                                            className="flex items-start gap-[5px] break-all text-[16px] leading-[24px] text-primary mb-[5px] focus:shadow-none focus:text-blue"
                                                            href={`mailto:${getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'email'
                                                            )}`}
                                                        >
                                                            <i className="tio-email text-[15px]"></i>
                                                            <span className="flex-1">
                                                                {getApplicantValueByKey(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'email'
                                                                )}
                                                            </span>
                                                        </a>
                                                    )}
                                                    {getApplicantValueByKey(
                                                        jobSingleApplication?.meta_attributes,
                                                        'phone_number'
                                                    ) && (
                                                        <a
                                                            className="flex items-start gap-[5px] break-all text-[16px] leading-[24px] text-primary focus:shadow-none focus:text-blue"
                                                            href={`tel:${getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'phone_number'
                                                            )}`}
                                                        >
                                                            <i className="tio-call_talking_quiet text-[15px]"></i>
                                                            <span className="flex-1">
                                                                {getApplicantValueByKey(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'phone_number'
                                                                )}
                                                            </span>
                                                        </a>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                        {/* Preliminary Question */}
                                        {(getApplicantValueByKey(
                                            jobSingleApplication?.meta_attributes,
                                            'referred_to'
                                        ) ||
                                            getApplicantValueByKey(
                                                jobSingleApplication?.meta_attributes,
                                                'any_relation'
                                            )) && (
                                            <div className="mb-5 bg-white rounded p-[30px]">
                                                <div className="flex items-center gap-2.5 mb-4 pb-4 border-b border-border-mid text-primary">
                                                    <i className="tio-help_outlined text-[25px] leading-[1]"></i>
                                                    <h4 className="m-0 text-primary">
                                                        Preliminary Questions
                                                    </h4>
                                                </div>
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'referred_to'
                                                ) && (
                                                    <div className="border-b border-background mb-5 pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Were you referred to
                                                            this position by an
                                                            employee of this
                                                            company?
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment capitalize">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'referred_to'
                                                            )}
                                                        </div>
                                                        {getApplicantValueByKey(
                                                            jobSingleApplication?.meta_attributes,
                                                            'referred_person'
                                                        ) && (
                                                            <>
                                                                <div className="mt-2 text-b3-mid text-headline">
                                                                    Referred By:
                                                                </div>
                                                                <div>
                                                                    {nl2br(
                                                                        getApplicantValueByKey(
                                                                            jobSingleApplication?.meta_attributes,
                                                                            'referred_person'
                                                                        )
                                                                    )}
                                                                </div>
                                                            </>
                                                        )}
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'any_relation'
                                                ) && (
                                                    <div>
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Do you have any relation
                                                            to an existing employee
                                                            at this company?
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment capitalize">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'any_relation'
                                                            )}
                                                        </div>
                                                        {getApplicantValueByKey(
                                                            jobSingleApplication?.meta_attributes,
                                                            'relative_person'
                                                        ) && (
                                                            <>
                                                                <div className="mt-2 text-b3-mid text-headline">
                                                                    Relatives:
                                                                </div>
                                                                <div>
                                                                    {nl2br(
                                                                        getApplicantValueByKey(
                                                                            jobSingleApplication?.meta_attributes,
                                                                            'relative_person'
                                                                        )
                                                                    )}
                                                                </div>
                                                            </>
                                                        )}
                                                    </div>
                                                )}
                                            </div>
                                        )}
                                        {/* Schedule & Salary Information */}
                                        <div className="mb-5 bg-white rounded p-[30px]">
                                            <div className="flex items-center gap-2.5 mb-4 pb-4 border-b border-border-mid text-primary">
                                                <i className="tio-money text-[25px] leading-[1]"></i>
                                                <h4 className="m-0 text-primary">
                                                    Schedule & Salary Information
                                                </h4>
                                            </div>
                                            <div className="grid grid-cols-3 gap-5">
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'work_experience'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Working Experience ( In
                                                            Year)
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'work_experience'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'about_us'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            How Did You Hear About
                                                            Us
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'about_us'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'when_join'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Joining Notice Period
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'when_join'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'expected_salary'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Expected Salary
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {`${
                                                                settings?.currency_symbol
                                                            }${formatSalaryAmount(
                                                                getApplicantValueByKey(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'expected_salary'
                                                                )
                                                            )}`}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'current_salary'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Current Salary
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {`${
                                                                settings.currency_symbol
                                                            }${formatSalaryAmount(
                                                                getApplicantValueByKey(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'current_salary'
                                                                )
                                                            )}`}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'education'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Education
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'education'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'education_institute'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Education Institute
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'education_institute'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'current_company_name'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                        Current Company Name
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            {getApplicantValueByKey(
                                                                jobSingleApplication?.meta_attributes,
                                                                'current_company_name'
                                                            )}
                                                        </div>
                                                    </div>
                                                )}
                                                {getApplicantValueByKey(
                                                    jobSingleApplication?.meta_attributes,
                                                    'protfolio_link'
                                                ) && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Portfolio Link
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            <a
                                                                href={getFormattedUrl(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'protfolio_link'
                                                                )}
                                                                target="_blank"
                                                                rel="noopener noreferrer"
                                                                className="text-blue"
                                                            >
                                                                {getApplicantValueByKey(
                                                                    jobSingleApplication?.meta_attributes,
                                                                    'protfolio_link'
                                                                )}
                                                            </a>
                                                        </div>
                                                    </div>
                                                )}
                                                {jobSingleApplication?.resume && (
                                                    <div className="border-b border-background pb-2.5">
                                                        <div className="text-b3-mid text-headline mb-[5px]">
                                                            Resume
                                                        </div>
                                                        <div className="text-b1-regular text-b1-notice-comment">
                                                            <button 
                                                                className="flex items-center gap-1 text-secondary font-bold cursor-pointer focus:shadow-none" 
                                                                onClick={() => pdfViewerModalHandler()} type="button">
                                                                See Attachment
                                                            </button>
                                                        </div>
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                        {/* More Information */}
                                        {(getApplicantValueByKey(
                                            jobSingleApplication?.meta_attributes,
                                            'explain_work_experience'
                                        ) ||
                                            getApplicantValueByKey(
                                                jobSingleApplication?.meta_attributes,
                                                'additional_comments'
                                            )) && (
                                            <div className="mb-5 bg-white rounded p-7 word-autophrase">
                                                <div className="flex items-center gap-2.5 mb-4 pb-4 border-b border-border-mid text-primary">
                                                    <i className="tio-file_text_outlined text-[25px] leading-[1]"></i>
                                                    <h4 className="m-0 text-primary">
                                                        More Information
                                                    </h4>
                                                </div>
                                                <div className="grid gap-5">
                                                    {getApplicantValueByKey(
                                                        jobSingleApplication?.meta_attributes,
                                                        'explain_work_experience'
                                                    ) && (
                                                        <div className="border-b border-background pb-2.5">
                                                            <div className="text-b3-mid text-headline mb-[5px]">
                                                                Working Experience
                                                            </div>
                                                            <div className="text-b1-regular text-b1-notice-comment">
                                                                {nl2br(
                                                                    getApplicantValueByKey(
                                                                        jobSingleApplication?.meta_attributes,
                                                                        'explain_work_experience'
                                                                    )
                                                                )}
                                                            </div>
                                                        </div>
                                                    )}
                                                    {getApplicantValueByKey(
                                                        jobSingleApplication?.meta_attributes,
                                                        'additional_comments'
                                                    ) && (
                                                        <div>
                                                            <div className="text-b3-mid text-headline mb-[5px]">
                                                                Additional Comment
                                                            </div>
                                                            <div className="text-b1-regular text-b1-notice-comment">
                                                                {nl2br(
                                                                    getApplicantValueByKey(
                                                                        jobSingleApplication?.meta_attributes,
                                                                        'additional_comments'
                                                                    )
                                                                )}
                                                            </div>
                                                        </div>
                                                    )}
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                    <div className="max-w-[490px] w-full">
                                        {/* Comments Area */}
                                        <div className="bg-white rounded p-[30px]">
                                            <div className="flex items-center gap-2.5 mb-5 pb-5 border-b border-border-mid text-primary">
                                                <i className="tio-comment_text_outlined text-[25px] leading-[1]"></i>
                                                <h4 className="m-0 text-primary">
                                                    Comments
                                                </h4>
                                            </div>
                                            {/* Comment Form */}
                                            <div className="pb-5 mb-5 border-b border-border-light">
                                                <div className="flex items-start gap-2.5">
                                                    {getUser?.avatar_url && (
                                                        <img
                                                            className="object-cover w-[42px] h-[42px] m-0 rounded-full"
                                                            src={
                                                                getUser?.avatar_url
                                                            }
                                                            alt={
                                                                getUser?.display_name
                                                            }
                                                        />
                                                    )}
                                                    <form
                                                        onSubmit={(e) =>
                                                            submitCommentHandler(e)
                                                        }
                                                        className="flex-1"
                                                    >
                                                        <Input
                                                            type="text"
                                                            name="comment"
                                                            placeholder="Write a comment..."
                                                            className="!pr-[30px]"
                                                            value={commentText}
                                                            onChange={(text) =>
                                                                setCommentText(
                                                                    text.value
                                                                )
                                                            }
                                                            errorMessage={
                                                                commentFormError
                                                            }
                                                            right={
                                                                <button
                                                                    type="submit"
                                                                    className="text-[18px] text-secondary absolute top-[5px] right-[15px]"
                                                                >
                                                                    <i className="tio-telegram"></i>
                                                                </button>
                                                            }
                                                        />
                                                    </form>
                                                </div>
                                            </div>
                                            {/* Comment List View */}
                                            {applicationComments?.map(
                                                (comment, index) => (
                                                    <div
                                                        className={
                                                            applicationComments?.length -
                                                                1 !==
                                                            index
                                                                ? 'border-b border-border-light pb-5 mb-5'
                                                                : ''
                                                        }
                                                        key={comment.id}
                                                    >
                                                        <div
                                                            style={{
                                                                backgroundColor:
                                                                    comment
                                                                        ?.app_process_obj
                                                                        ?.icon_color,
                                                            }}
                                                            className="inline-flex items-center gap-1.5 px-2.5 py-[2px] mb-5 text-white rounded-xsm text-[15px]"
                                                        >
                                                            <i
                                                                className={`${comment?.app_process_obj?.icon} text-[14px]`}
                                                            ></i>
                                                            {
                                                                comment
                                                                    ?.app_process_obj
                                                                    ?.name
                                                            }
                                                        </div>
                                                        <div className="flex gap-3 item-start">
                                                            <img
                                                                className="h-[42px] w-[42px] m-0 rounded-full object-cover"
                                                                src={
                                                                    comment
                                                                        ?.user_obj
                                                                        ?.avatar_url
                                                                }
                                                                alt={
                                                                    comment
                                                                        ?.user_obj
                                                                        ?.display_name
                                                                }
                                                            />
                                                            <div>
                                                                <div className="flex items-center gap-2.5 mb-1">
                                                                    <span className="text-b2-regular text-headline">
                                                                        {
                                                                            comment
                                                                                ?.user_obj
                                                                                ?.display_name
                                                                        }
                                                                    </span>
                                                                    <span className="text-[14px] text-b3-meta">
                                                                        {formatDateOnly(
                                                                            comment?.created_at
                                                                        )}
                                                                    </span>
                                                                </div>
                                                                <p className="m-0 text-b2-regular text-b1-notice-comment">
                                                                    {
                                                                        comment?.comment
                                                                    }
                                                                </p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                )
                                            )}
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </>
                )}
            </Modal>
            <PdfViewer pdfSrc={jobSingleApplication?.resume} isPdfViewModalOpen={isPdfViewModalOpen} closePdfViewModal={closePdfViewModal} />
        </>
    );
};

export default ApplicationPreviewModal;
