/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import ReactPaginate from 'react-paginate';

interface IPagination {
    /**
     * How many items per page.
     */
    perPage?: number;

    /**
     * Total items.
     */
    total?: number | any;

    /**
     * Current page.
     */
    currentPage?: number;

    /**
     * Paginate handler.
     */
    paginate?: Function;
}

const Pagination = (props: IPagination) => {
    const {
        perPage = 10,
        total = 1,
        paginate = () => {},
        currentPage = 1,
    } = props;
    const [current, setCurrent] = useState(currentPage);
    const [totalPage, setTotalPage] = useState(Math.ceil(total / perPage));
    const [initialRender, setInitialRender] = useState(true);

    const containerClass = `list-none inline-flex item-center overflow-hidden m-0 p-0 border border-border-mid rounded-sm`;
    const pageClass = `m-0 p-0 bg-shadebg border-r border-border-mid last:border-r-0`;
    const pageActiveClass = ``;
    const pageLinkClass = `h-[44px] w-[44px] flex items-center justify-center px-2 text-[14px] font-medium text-headline transition-all hover:!text-white hover:!bg-secondary focus:shadow-none focus:outline-none focus:!bg-secondary focus:!text-white`;
    const pageLinkActiveClass = `!bg-secondary !text-white`;
    const disableClass = ``;
    const disableLinkClass = `opacity-10 pointer-events-none focus:bg-white focus:text-headline`;

    useEffect(() => {
        const newTotalPage = Math.ceil(total / perPage);
        setTotalPage(newTotalPage);

        if (initialRender) {
            setInitialRender(false);
        } else {
            // Reset current page to 1 if data changes and it's not the initial render
            setCurrent(1);
            paginate(null);
        }
    }, [total, perPage]);

    useEffect(() => {
        setCurrent(currentPage);
    }, [currentPage]);

    return (
        <>
            {totalPage > 1 && (
                <div className="ai_job_listing-pagination pt-8">
                    <ReactPaginate
                        pageCount={totalPage}
                        onPageChange={(page) => {
                            const newPage = page.selected + 1;
                            paginate(newPage);
                            setCurrent(newPage);
                        }}
                        initialPage={current - 1}
                        previousLabel={<i className="tio-back_ui" />}
                        nextLabel={<i className="tio-next_ui" />}
                        breakLabel={'...'}
                        breakClassName={pageClass}
                        breakLinkClassName={pageLinkClass}
                        marginPagesDisplayed={2}
                        pageRangeDisplayed={2}
                        containerClassName={containerClass}
                        activeClassName={pageActiveClass}
                        pageClassName={pageClass}
                        pageLinkClassName={pageLinkClass}
                        activeLinkClassName={pageLinkActiveClass}
                        nextClassName={pageClass}
                        nextLinkClassName={pageLinkClass}
                        previousClassName={pageClass}
                        previousLinkClassName={pageLinkClass}
                        disabledClassName={disableClass}
                        disabledLinkClassName={disableLinkClass}
                        disableInitialCallback
                    />
                </div>
            )}
        </>
    );
};

export default Pagination;
