/**
 * External dependencies
 */
import { Children } from '@wordpress/element';
import { twMerge } from 'tailwind-merge';

type Props = {
    header: Array<any>;
    body: Array<any>;
    children?: React.ReactNode;
};

const Table = ({ header, body, children }: Props) => {
    // Create a mapping of children based on the 'customKey' prop in the parent div
    const customColumns = Children.toArray(children).reduce(
        (acc: any, child: any) => {
            if (child.props && child.props.customKey) {
                acc[child.props.customKey] = child;
            }
            return acc;
        },
        {}
    );

    const renderCell = (key: string, rowData: any) => {
        const customColumn = customColumns[key];
        if (customColumn) {
            // Check if the child has a function as children
            const childContent = customColumn.props.children;
            if (typeof childContent === 'function') {
                // Call the function and pass rowData to it
                return childContent(rowData);
            }
        }
        // Check explicitly for undefined or null to display 0 or other falsy values
        return rowData[key] !== undefined && rowData[key] !== null
            ? rowData[key]
            : '';
    };

    return (
        <div className="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th></th>
                        {header.map((head, headIndex) => (
                            <th key={headIndex} className={twMerge(head.class)}>
                                {head.title}
                            </th>
                        ))}
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    {body?.length ? (
                        body.map((row, rowIndex) => (
                            <tr key={rowIndex}>
                                <td></td>
                                {header.map((head, colIndex) => (
                                    <td
                                        key={colIndex}
                                        className={twMerge(head.bodyClass)}
                                    >
                                        {renderCell(head.key, row)}
                                    </td>
                                ))}
                                <td></td>
                            </tr>
                        ))
                    ) : (
                        <tr>
                            <td></td>
                            <td colSpan={header.length}>
                                <h5 className="text-center p-5 mb-0">
                                    No Data Found
                                </h5>
                            </td>
                            <td></td>
                        </tr>
                    )}
                </tbody>
            </table>
        </div>
    );
};

export default Table;
