/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */

import { IJobListing } from './state';
import * as Types from '../types';
import { SwalError, SwalSuccess } from '../../utils/Swal';
import { prepareDataForApplyForm } from '../../utils/Global';
import {
    jobsApplyEndpoint,
    attributeValueJobCountsFrontendEndpoint,
    jobsFrontendEndpoint,
} from '../endpoint';
import { IResponseGenerator } from '../../interfaces';

const actions = {
    setJobs(jobs: IJobListing) {
        return {
            type: Types.SET_CAREER_JOBS,
            jobs,
        };
    },
    setJobsLoading(jobsLoading: boolean) {
        return {
            type: Types.SET_CAREER_JOBS_LOADING,
            jobsLoading,
        };
    },
    setJobsTotalCount(jobsTotalCount: number) {
        return {
            type: Types.SET_CAREER_JOBS_TOTAL,
            jobsTotalCount,
        };
    },
    setJobsTotalPage(jobsTotalPage: number) {
        return {
            type: Types.SET_CAREER_JOBS_TOTAL_PAGE,
            jobsTotalPage,
        };
    },
    setCvPostForm(cvPostForm: IJobListing) {
        return {
            type: Types.SET_CV_POST_FORM,
            cvPostForm,
        };
    },
    setCvPostFormFields(cvPostFormFields: IJobListing) {
        return {
            type: Types.SET_CV_POST_FORM_FIELDS,
            cvPostFormFields,
        };
    },
    setCvFormStep(cvFormStep: number) {
        return {
            type: Types.SET_CV_FORM_STEP,
            cvFormStep,
        };
    },
    setCvFormStepCompleted(cvFormStepCompleted: number) {
        return {
            type: Types.SET_CV_FORM_STEP_COMPLETED,
            cvFormStepCompleted,
        };
    },
    setCvFormLoading(cvFormLoading: boolean) {
        return {
            type: Types.SET_CV_FORM_LOADING,
            cvFormLoading,
        };
    },
    setSidebarFilterData(sidebarFilterData: IJobListing) {
        return {
            type: Types.SET_SIDEBAR_FILTER_DATA,
            sidebarFilterData,
        };
    },
    setSidebarFilterLoading(sidebarFilterLoading: boolean) {
        return {
            type: Types.SIDEBAR_FILTER_LOADING,
            sidebarFilterLoading,
        };
    },
    setCareerPageSettings(careerPageSettings: IJobListing) {
        return {
            type: Types.SET_CAREER_PAGE_SETTINGS,
            careerPageSettings,
        };
    },
    setCareerPageSettingsLoading(careerPageSettingsLoading: boolean) {
        return {
            type: Types.SET_CAREER_PAGE_SETTINGS_LOADING,
            careerPageSettingsLoading,
        };
    },
    fetchFromAPI(path: string) {
        return {
            type: Types.FETCH_FROM_API,
            path,
        };
    },
    fetchFromAPIUnparsed(path: string) {
        return {
            type: Types.FETCH_FROM_API_UNPARSED,
            path,
        };
    },
    *getJobs(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            yield actions.setJobsLoading(true);
            const response: IResponseGenerator =
                yield actions.fetchFromAPIUnparsed(`${jobsFrontendEndpoint}?${params}`);
            let totalCount = 0;
            let totalPage = 0;

            if (response.headers !== undefined) {
                totalPage = response.headers.get('X-WP-TotalPages');
                totalCount = response.headers.get('X-WP-Total');
            }

            // console.log(totalCount);
            yield actions.setJobsTotalCount(totalCount);
            yield actions.setJobsTotalPage(totalPage);
            yield actions.setJobs(response.data);
            yield actions.setJobsLoading(false);
            return response.data;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Job List', error);
        }
    },
    *saveCvPostForm(payload: any) {
        yield actions.setCvFormLoading(true);
        const prepareMeta = prepareDataForApplyForm(payload.data);
        const applyDataForPost: any = {
            job_id: payload.job_id,
            _wpnonce: payload._wpnonce,
            app_meta: prepareMeta,
        };

        // Create a new FormData object
        const formData = new FormData();

        for (const key in applyDataForPost) {
            if (
                typeof applyDataForPost[key] === 'object' &&
                applyDataForPost[key] !== null
            ) {
                for (const subKey in applyDataForPost[key]) {
                    if (
                        typeof applyDataForPost[key][subKey] === 'object' &&
                        applyDataForPost[key][subKey] !== null
                    ) {
                        for (const subSubKey in applyDataForPost[key][subKey]) {
                            formData.append(
                                `${key}[${subKey}][${subSubKey}]`,
                                applyDataForPost[key][subKey][subSubKey]
                            );
                        }
                    } else {
                        formData.append(
                            `${key}[${subKey}]`,
                            applyDataForPost[key][subKey]
                        );
                    }
                }
            } else {
                formData.append(key, applyDataForPost[key]);
            }
        }

        try {
            let response: any = {};
            if (payload.id) {
                response = yield {
                    type: Types.UPDATE_CV_POST_FORM,
                    payload: formData,
                    path: `${jobsApplyEndpoint}/${payload.id}`,
                };
            } else {
                response = yield {
                    type: Types.CREATE_CV_POST_FORM,
                    payload: formData,
                    path: jobsApplyEndpoint,
                };
            }
            if (response) {
                SwalSuccess({
                    title: __('Done', 'ai-job-listing'),
                    text: 'Resume Submitted Successfuly',
                });
            }
            yield actions.setCvPostForm({});
            yield actions.setCvFormLoading(false);
            return response;
        } catch (error: any) {
            yield actions.setCvFormLoading(false);
            SwalError({
                title: __('Errors', 'ai-job-listing'),
                text: error.message,
                toast: true,
                position: 'bottom',
            });
        }
    },
    *getSidebarFilterData(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            const path = `${attributeValueJobCountsFrontendEndpoint}?${params}`;
            const response = yield actions.fetchFromAPI(path);
            if (response) {
                const getAllDepartment = response.filter(
                    (department: any) =>
                        department.attribute_slug === 'job-departments'
                );
                const getAllJobTypes = response.filter(
                    (department: any) =>
                        department.attribute_slug === 'job-types'
                );
                const getAllJobShift = response.filter(
                    (department: any) =>
                        department.attribute_slug === 'job-shifts'
                );
                const getAllJobLocation = response.filter(
                    (department: any) =>
                        department.attribute_slug === 'job-locations'
                );
                const settings: any = {
                    jobTypes: [
                        {
                            attribute_value_id: '',
                            attribute_value: 'All',
                            attribute_slug: 'job-types',
                            id: '',
                        },
                        ...getAllJobTypes,
                    ],
                    jobShifts: [
                        {
                            attribute_value_id: '',
                            attribute_value: 'All',
                            attribute_slug: 'job-shifts',
                            id: '',
                        },
                        ...getAllJobShift,
                    ],
                    departments: [
                        {
                            attribute_value_id: '',
                            attribute_value: 'All',
                            attribute_slug: 'job-departments',
                            id: '',
                        },
                        ...getAllDepartment,
                    ],
                    locations: [
                        {
                            attribute_value_id: '',
                            attribute_value: 'All',
                            attribute_slug: 'job-locations',
                            id: '',
                        },
                        ...getAllJobLocation,
                    ],
                };
                yield actions.setSidebarFilterData(settings);
            }
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Listing Sidebar Data', error);
        }
    },
};

export default actions;
