/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */
import { IGeneralSettings } from './state';
import * as Types from '../types';
import { SwalError, SwalSuccess } from '../../utils/Swal';
import {
    updateSettingsEndpoint,
    emailTemplatesEndpoint,
    generalSettingEndpoint,
    generalSettingsEndpoint,
    searchPagesEndpoint,
} from '../endpoint';
import { IResponseGenerator } from '../../interfaces';

const actions = {
    setGeneralSettingsField(generalSettingsField: IGeneralSettings) {
        return {
            type: Types.SET_GENERAL_SETTINGS_FIELD,
            generalSettingsField,
        };
    },
    setGeneralSettings(generalSettings: IGeneralSettings) {
        return {
            type: Types.SET_GENERAL_SETTINGS,
            generalSettings,
        };
    },
    setGeneralSettingsLoading(generalSettingsLoading: boolean) {
        return {
            type: Types.SET_GENERAL_SETTINGS_LOADING,
            generalSettingsLoading,
        };
    },
    setGeneralSettingSaveLoading(generalSettingSaveLoading: boolean) {
        return {
            type: Types.SET_GENERAL_SETTING_SAVE_LOADING,
            generalSettingSaveLoading,
        };
    },
    setEmailTemplates(emailTemplates: IGeneralSettings) {
        return {
            type: Types.SET_EMAIL_TEMPLATES,
            emailTemplates,
        };
    },
    setEmailTemplatesLoading(emailTemplatesLoading: boolean) {
        return {
            type: Types.SET_EMAIL_TEMPLATES_LOADING,
            emailTemplatesLoading,
        };
    },
    setEmailTemplateDetailsLoading(emailTemplateDetailsLoading: boolean) {
        return {
            type: Types.SET_EMAIL_TEMPLATE_DETAILS_LOADING,
            emailTemplateDetailsLoading,
        };
    },
    setEmailTemplateSaveLoading(emailTemplateSaveLoading: boolean) {
        return {
            type: Types.SET_EMAIL_TEMPLATE_SAVE_LOADING,
            emailTemplateSaveLoading,
        };
    },
    setCurrentUser(currentUser: IGeneralSettings) {
        return {
            type: Types.SET_CURRENT_USER,
            currentUser,
        };
    },
    fetchFromAPI(path: string) {
        return {
            type: Types.FETCH_FROM_API,
            path,
        };
    },
    fetchFromAPIUnparsed(path: string) {
        return {
            type: Types.FETCH_FROM_API_UNPARSED,
            path,
        };
    },
    *getGeneralSettingsField() {
        try {
            yield actions.setGeneralSettingsLoading(true);
            const response = yield actions.fetchFromAPI(
                generalSettingsEndpoint
            );
            if (response) {
                yield actions.setGeneralSettingsField(response);
            }
            yield actions.setGeneralSettingsLoading(false);
            return response;
        } catch (error) {
            yield actions.setGeneralSettingsLoading(false);
            // eslint-disable-next-line no-console
            console.error('Error Fetching Global Settings Field', error);
        }
    },
    *getGeneralSettings() {
        try {
            yield actions.setGeneralSettingsLoading(true);
            const response = yield actions.fetchFromAPI(generalSettingEndpoint);
            if (response) {
                yield actions.setGeneralSettings({
                    ...response,
                    posts_per_page: 10,
                });
            }
            yield actions.setGeneralSettingsLoading(false);
        } catch (error) {
            yield actions.setGeneralSettingsLoading(false);
            // eslint-disable-next-line no-console
            console.error('Error Fetching Global Settings', error);
        }
    },
    *updateGeneralSettings(payload: any) {
        yield actions.setGeneralSettingSaveLoading(true);
        try {
            let response: any = {};
            response = yield {
                type: Types.UPDATE_GENERAL_SETTINGS,
                payload,
                path: updateSettingsEndpoint,
            };
            yield actions.setGeneralSettingSaveLoading(false);
            return response;
        } catch (error: any) {
            yield actions.setGeneralSettingSaveLoading(false);
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },

    *getEmailTemplateDetails(payload: any) {
        yield actions.setEmailTemplateDetailsLoading(true);
        try {
            const response = yield actions.fetchFromAPI(
                `${emailTemplatesEndpoint}/${payload}`
            );
            if (response) {
                const setTemplateToForm: any = {
                    id: response?.id,
                    has_notification_status: response?.has_notification_status,
                    subject: response?.subject,
                    message: response?.message,
                    title: response?.type?.name,
                };
                yield actions.setEmailTemplateDetailsLoading(false);
                return setTemplateToForm;
            }
        } catch (error: any) {
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *saveEmailTemplate(payload: any) {
        yield actions.setEmailTemplateSaveLoading(true);
        try {
            let response: IResponseGenerator = {};
            if (payload.id) {
                response = yield {
                    type: Types.UPDATE_GENERAL_SETTINGS,
                    payload,
                    path: `${emailTemplatesEndpoint}/${payload.id}`,
                };
            }

            yield actions.setEmailTemplateSaveLoading(false);
            if (response.id == payload.id) {
                SwalSuccess({
                    title: __('Updated', 'ai-job-listing'),
                    text: 'Email Template Updated Successfuly',
                });
            }
        } catch (error: any) {
            yield actions.setEmailTemplateSaveLoading(false);
            SwalError({
                title: __('Errors', 'ai-job-listing'),
                text: error.message,
                toast: true,
                position: 'bottom',
            });
        }
    },

    *getCareerPage(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();
        try {
            const response = yield actions.fetchFromAPI(`${searchPagesEndpoint}?${params}`);
            return response.pages;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Pages', error);
        }
    },
};

export default actions;
