/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */
import {
    jobsEndpoint,
    jobStatusCountEndpoint,
    jobUpdateItemEndpoint,
    jobsRestoreEndpoint,
    jobsDeleteEndpoint,
    jobsApplicationEndpoint,
    jobsApplicationStatusCountEndpoint,
    applicationItemUpdateEndpoint,
    processCommentUpdateEndpoint,
    getJobAiDescriptionEndpoint,
    ApplicationReportEndpoint,
    listOfUsersByJobIdEndpoint,
    updateJobNotificationEmailsEndpoint,
    allUsersWithRolesEndpoint,
    jobsApplyEndpoint,
    hasJobLimitEndpoint,
} from '../endpoint';
import { IJobPostForm, IJob } from './state';
import * as Types from '../types';
import { SwalError, SwalSuccess } from '../../utils/Swal';
import { IResponseGenerator } from '../../interfaces';
import { getIdByKey } from '../../utils/Global';

const actions = {
    setJobPostForm(jobPostForm: IJobPostForm) {
        return {
            type: Types.SET_JOB_POST_FORM,
            jobPostForm,
        };
    },
    setJobAiDescription(jobAiDescription: IJob) {
        return {
            type: Types.SET_JOB_AI_DESCRIPTION,
            jobAiDescription,
        };
    },
    setJobAiDescriptionLoading(jobAiDescriptionLoading: boolean) {
        return {
            type: Types.SET_JOB_AI_DESCRIPTION_LOADING,
            jobAiDescriptionLoading,
        };
    },
    setJobPostFormLoading(jobPostFormLoading: boolean) {
        return {
            type: Types.SET_JOB_POST_FORM_LOADING,
            jobPostFormLoading,
        };
    },
    setJobPostFormError(jobPostFormError: boolean) {
        return {
            type: Types.SET_JOB_POST_FORM_ERROR,
            jobPostFormError,
        };
    },
    setJobFormStep(jobFormStep: number) {
        return {
            type: Types.SET_JOB_FORM_STEP,
            jobFormStep,
        };
    },
    setJobFormStepCompleted(jobFormStepCompleted: number) {
        return {
            type: Types.SET_JOB_FORM_STEP_COMPLETED,
            jobFormStepCompleted,
        };
    },
    setJobLists(jobLists: IJobPostForm) {
        return {
            type: Types.SET_JOB_LISTS,
            jobLists,
        };
    },
    setJob(job: IJob) {
        return {
            type: Types.SET_JOB,
            job,
        };
    },
    setHasJobLimit(hasJobLimit: boolean) {
        return {
            type: Types.SET_AXILWEB_AJL_JOB_LIMIT,
            hasJobLimit,
        };
    },
    setJobStatusCount(jobStatusCount: IJob) {
        return {
            type: Types.SET_JOB_STATUS_COUNT,
            jobStatusCount,
        };
    },
    setJobListsLoading(jobListsLoading: boolean) {
        return {
            type: Types.SET_JOB_LISTS_LOADING,
            jobListsLoading,
        };
    },
    setJobDetailsLoading(jobDetailsLoading: boolean) {
        return {
            type: Types.SET_JOB_DETAILS_LOADING,
            jobDetailsLoading,
        };
    },
    setJobListsTotalCount(jobListsTotalCount: number) {
        return {
            type: Types.SET_JOB_LISTS_TOTAL_COUNT,
            jobListsTotalCount,
        };
    },
    setJobListsTotalPages(jobListsTotalPages: number) {
        return {
            type: Types.SET_JOB_LISTS_TOTAL_PAGES,
            jobListsTotalPages,
        };
    },
    setJobApplicationList(jobApplicationList: IJob) {
        return {
            type: Types.SET_JOB_APPLICATION_LIST,
            jobApplicationList,
        };
    },
    setJobApplicationStatusCount(jobApplicationStatusCount: IJob) {
        return {
            type: Types.SET_JOB_APPLICATION_STATUS_COUNT,
            jobApplicationStatusCount,
        };
    },
    setJobApplicationLoading(jobApplicationLoading: boolean) {
        return {
            type: Types.SET_JOB_APPLICATION_LOADING,
            jobApplicationLoading,
        };
    },
    setJobApplicationTotalCount(jobApplicationTotalCount: number) {
        return {
            type: Types.SET_JOB_APPLICATION_TOTAL_COUNT,
            jobApplicationTotalCount,
        };
    },
    setJobApplicationTotalPage(jobApplicationTotalPage: number) {
        return {
            type: Types.SET_JOB_APPLICATION_TOTAL_PAGE,
            jobApplicationTotalPage,
        };
    },
    setJobApplication(jobApplication: IJob) {
        return {
            type: Types.SET_JOB_APPLICATION,
            jobApplication,
        };
    },
    setJobApplicationDetailsLoading(jobApplicationDetailsLoading: boolean) {
        return {
            type: Types.SET_JOB_APPLICATION_DETAILS_LOADING,
            jobApplicationDetailsLoading,
        };
    },
    setApplicationComments(jobApplicationComments: IJob) {
        return {
            type: Types.SET_APPLICATION_COMMENTS,
            jobApplicationComments,
        };
    },
    setApplicationCommentsLoading(jobApplicationCommentsLoading: boolean) {
        return {
            type: Types.SET_APPLICATION_COMMENTS_LOADING,
            jobApplicationCommentsLoading,
        };
    },
    setApplicationCommentCounts(jobApplicationCommentCounts: number) {
        return {
            type: Types.SET_APPLICATION_COMMENT_COUNTS,
            jobApplicationCommentCounts,
        };
    },
    setApplicationCommentPages(jobApplicationCommentPages: number) {
        return {
            type: Types.SET_APPLICATION_COMMENT_PAGES,
            jobApplicationCommentPages,
        };
    },
    setJobApplicationDates(jobApplicationDates: IJob) {
        return {
            type: Types.SET_JOB_APPLICATION_DATES,
            jobApplicationDates,
        };
    },
    setJobApplicationDatesLoading(jobApplicationDatesLoading: boolean) {
        return {
            type: Types.SET_JOB_APPLICATION_DATES_LOADING,
            jobApplicationDatesLoading,
        };
    },
    setUsersEmailList(usersEmailList: IJob) {
        return {
            type: Types.SET_USERS_EMAIL_LISTS,
            usersEmailList,
        };
    },
    setMailNotificationUsersByJob(mailNotificationUsersByJob: IJob) {
        return {
            type: Types.SET_MAIL_NOTIFICATION_USERS_BY_JOB,
            mailNotificationUsersByJob,
        };
    },
    setLoadingMailNotificationUsers(loadingMailNotificationUsers: boolean) {
        return {
            type: Types.SET_LOADING_MAIL_NOTIFICATION_USERS,
            loadingMailNotificationUsers,
        };
    },
    setJobOverviewStatus(jobOverviewStatus: IJob) {
        return {
            type: Types.SET_JOB_OVERVIEW_STATUS,
            jobOverviewStatus,
        };
    },
    setJobOverviewStatusLoading(jobOverviewStatusLoading: boolean) {
        return {
            type: Types.SET_JOB_OVERVIEW_STATUS_LOADING,
            jobOverviewStatusLoading,
        };
    },
    fetchFromAPI(path: string) {
        return {
            type: Types.FETCH_FROM_API,
            path,
        };
    },
    fetchFromAPIUnparsed(path: string) {
        return {
            type: Types.FETCH_FROM_API_UNPARSED,
            path,
        };
    },
    *getHasJobLimit() {
        const response = yield actions.fetchFromAPI(hasJobLimitEndpoint);
        yield actions.setHasJobLimit(response);
        return response;
    },
    *saveJobPostForm(payload: IJobPostForm) {
        yield actions.setJobPostFormLoading(true);
        yield actions.setJobPostFormError(false);
        const preparePayload = {
            ...payload,
        };
        try {
            let response: IJobPostForm = {};
            if (payload.id) {
                response = yield {
                    type: Types.UPDATE_JOB_POST,
                    payload: preparePayload,
                    path: `${jobsEndpoint}/${payload.id}`,
                };
            } else {
                response = yield {
                    type: Types.CREATE_JOB_POST,
                    payload: preparePayload,
                    path: jobsEndpoint,
                };
            }
            if (response.id) {
                yield actions.getJobPostForm(response.id.toString());
            }
            yield actions.setJobPostFormLoading(false);
            return response;
        } catch (error: any) {
            yield actions.setJobPostFormLoading(false);
            yield actions.setJobPostFormError(true);
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *generateAiDescription(payload: any) {
        yield actions.setJobAiDescriptionLoading(true);
        try {
            const response = yield {
                type: 'CREATE_OR_UPDATE_API',
                payload,
                path: getJobAiDescriptionEndpoint,
            };
            yield actions.setJobAiDescription(response);
            yield actions.setJobAiDescriptionLoading(false);
            return response;
        } catch (error) {
            yield actions.setJobAiDescriptionLoading(false);
            console.error('Error gerating AI description:', error);
        }
    },
    *getJobPostForm(payload: string) {
        const queryParams = {
            id: payload,
        };
        const params = new URLSearchParams(queryParams);
        yield actions.setJobPostFormLoading(true);
        try {
            const path = `${jobsEndpoint}?${params}`;
            const response = yield actions.fetchFromAPI(path);
            const responseData = response[0];

            if (responseData) {
                yield actions.setJobPostForm({
                    ...responseData,
                    job_departments:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_departments'
                        ) ?? null,
                    job_shifts:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_shifts'
                        ) ?? null,
                    job_types:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_types'
                        ) ?? null,
                    job_locations:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_locations'
                        ) ?? null,
                    job_salary_type:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_salary_type'
                        ) ?? null,
                    job_education_qualifications:
                        getIdByKey(
                            responseData.attribute_values,
                            'job_education_qualifications'
                        ) ?? null,
                    experienced_year: responseData.experienced_year?.id,
                    experienced_year_label:
                        responseData.experienced_year?.value,
                });
                if (Number(responseData.form_step_complete) < 5) {
                    yield actions.setJobFormStepCompleted(
                        Number(responseData.form_step_complete) + 1
                    );
                } else {
                    yield actions.setJobFormStepCompleted(
                        Number(responseData.form_step_complete)
                    );
                }
            }
            yield actions.setJobPostFormLoading(false);
            return responseData;
        } catch (error) {
            console.error('Error fetching single Job:', error);
        }
    },
    *getJobLists(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            yield actions.setJobListsLoading(true);
            const response: IResponseGenerator =
                yield actions.fetchFromAPIUnparsed(`${jobsEndpoint}?${params}`);
            let totalCount = 0;
            let totalPage = 0;
            if (response.headers !== undefined) {
                totalPage = response.headers.get('X-WP-TotalPages');
                totalCount = response.headers.get('X-WP-Total');
            }

            yield actions.setJobListsTotalCount(totalCount);
            yield actions.setJobListsTotalPages(totalPage);
            yield actions.setJobLists(response.data);
            yield actions.setJobListsLoading(false);
            return response.data;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Job List', error);
        }
    },
    *setJobDetails(payload: any) {
        if (payload === undefined) {
            payload = {};
        }
        const params = new URLSearchParams(
            payload as URLSearchParams
        ).toString();
        try {
            yield actions.setJobDetailsLoading(true);
            const path = `${jobsEndpoint}?${params}`;
            const response = yield actions.fetchFromAPI(path);
            const responseData = response[0];

            if (responseData?.id == payload.id) {
                yield actions.setJob({
                    ...responseData,
                });
            }
            yield actions.setJobDetailsLoading(false);
            return responseData;
        } catch (error) {
            console.error('Error fetching Job Details:', error);
        }
    },
    *getJobStatusCount() {
        try {
            const path = jobStatusCountEndpoint;
            const response = yield actions.fetchFromAPI(path);

            if (response) {
                yield actions.setJobStatusCount(response);
            }
            return response;
        } catch (error) {
            console.error('Error fetching Job Status Count:', error);
        }
    },
    *deleteJobPost(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.DELETE_JOB_POST,
                payload,
                path: jobsEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Moved', 'ai-job-listing'),
                    text: __('Job Move to Trash', 'ai-job-listing'),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Delete Job Post', error);
        }
    },
    *deletePermanentJobPost(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.DELETE_JOB_POST,
                payload,
                path: jobsDeleteEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Deleted', 'ai-job-listing'),
                    text: __('Deleted Job Permanently', 'ai-job-listing'),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Permanent Delete Job Post', error);
        }
    },
    *updateJobField(payload: IJobPostForm) {
        try {
            let response: IJobPostForm = {};
            response = yield {
                type: Types.UPDATE_JOB_POST,
                payload,
                path: jobUpdateItemEndpoint,
            };
            return response;
        } catch (error: any) {
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *restoreTrashJob(payload: any) {
        try {
            let response: IResponseGenerator = {};
            response = yield {
                type: Types.UPDATE_JOB_POST,
                payload,
                path: jobsRestoreEndpoint,
            };

            if (response) {
                SwalSuccess({
                    title: __('Restored', 'ai-job-listing'),
                    text: __('Job Restored Successfully', 'ai-job-listing'),
                    position: 'center',
                });
            }
            return response;
        } catch (error: any) {
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *setJobApplications(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            yield actions.setJobApplicationLoading(true);
            const response: IResponseGenerator =
                yield actions.fetchFromAPIUnparsed(
                    `${jobsApplicationEndpoint}?${params}`
                );
            let totalCount = 0;
            let totalPage = 0;
            if (response.headers !== undefined) {
                totalPage = response.headers.get('X-WP-TotalPages');
                totalCount = response.headers.get('X-WP-Total');
            }

            // console.log(totalCount);
            yield actions.setJobApplicationTotalCount(totalCount);
            yield actions.setJobApplicationTotalPage(totalPage);
            yield actions.setJobApplicationList(response.data);
            yield actions.setJobApplicationLoading(false);
            return response.data;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Job Application', error);
        }
    },
    *deleteJobApplication(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.DELETE_JOB_POST,
                payload,
                path: jobsApplyEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Moved', 'ai-job-listing'),
                    text: __('Application Moved to Trash', 'ai-job-listing'),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Delete Job Post', error);
        }
    },
    *setJobApplicationStatus(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();
        try {
            yield actions.setJobApplicationLoading(true);
            const response: any = yield actions.fetchFromAPI(
                `${jobsApplicationStatusCountEndpoint}?${params}`
            );
            if (response) {
                yield actions.setJobApplicationStatusCount(response);
            }
            yield actions.setJobApplicationLoading(false);
        } catch (error) {
            console.error(
                'Error fetching Job Application Status Count:',
                error
            );
        }
    },
    *updateJobApplicationStatus(payload: any) {
        try {
            let response: any = {};
            response = yield {
                type: Types.UPDATE_JOB_POST,
                payload,
                path: applicationItemUpdateEndpoint,
            };
            return response;
        } catch (error: any) {
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *setJobApplicationDetails(payload: any) {
        if (payload === undefined) {
            payload = {};
        }
        const params = new URLSearchParams(
            payload as URLSearchParams
        ).toString();

        try {
            yield actions.setJobApplicationDetailsLoading(true);
            const response: any = yield actions.fetchFromAPI(
                `${jobsApplicationEndpoint}?${params}`
            );
            const responseData = response[0];
            if (responseData?.id == payload.id) {
                yield actions.setJobApplication({
                    ...responseData,
                });
            }
            yield actions.setJobApplicationDetailsLoading(false);
            return responseData;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Job Application Details', error);
        }
    },
    *saveApplicationComment(payload: any) {
        try {
            let response: IJobPostForm = {};
            response = yield {
                type: Types.CREATE_JOB_POST,
                payload,
                path: processCommentUpdateEndpoint,
            };
            return response;
        } catch (error: any) {
            // eslint-disable-next-line no-console
            console.error('Error to insert comment', error);
        }
    },
    *setApplicationCommentById(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            yield actions.setApplicationCommentsLoading(true);
            const response: IResponseGenerator =
                yield actions.fetchFromAPIUnparsed(
                    `${processCommentUpdateEndpoint}?${params}`
                );
            let totalCount = 0;
            let totalPage = 0;
            if (response.headers !== undefined) {
                totalPage = response.headers.get('X-WP-TotalPages');
                totalCount = response.headers.get('X-WP-Total');
            }

            // console.log(totalCount);
            yield actions.setApplicationCommentCounts(totalCount);
            yield actions.setApplicationCommentPages(totalPage);
            yield actions.setApplicationComments(response.data);
            yield actions.setApplicationCommentsLoading(false);
            return response.data;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Comments', error);
        }
    },
    *setJobApplicationDateList(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();
        try {
            yield actions.setJobApplicationDatesLoading(true);
            const response: any = yield actions.fetchFromAPI(
                `${ApplicationReportEndpoint}?${params}`
            );
            if (response) {
                yield actions.setJobApplicationDates(response);
            }
            yield actions.setJobApplicationDatesLoading(false);
        } catch (error) {
            console.error('Error fetching Job Application Dates:', error);
        }
    },
    *getMailNotificationUsersByJob(id: any) {
        yield actions.setLoadingMailNotificationUsers(true);
        try {
            const response: any = yield actions.fetchFromAPI(
                `${listOfUsersByJobIdEndpoint}/${id}`
            );

            if (response) {
                yield actions.setMailNotificationUsersByJob(response);
            }
            yield actions.setLoadingMailNotificationUsers(false);
            return response;
        } catch (error) {
            yield actions.setLoadingMailNotificationUsers(false);
            console.error('Error fetching Users', error);
        }
    },
    *updateJobNotificationEmail(payload: any) {
        const prepareData = {
            job_notification_emails: payload.job_notification_emails,
        };
        try {
            let response: any = {};
            response = yield {
                type: Types.UPDATE_JOB_POST,
                payload: prepareData,
                path: `${updateJobNotificationEmailsEndpoint}/${payload.job_id}`,
            };
            return response;
        } catch (error: any) {
            // eslint-disable-next-line no-console
            SwalError({
                title: __('Error', 'ai-job-listing'),
                text: error.message,
                position: 'center',
            });
        }
    },
    *getUsersEmailList(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();
        try {
            const response: any = yield actions.fetchFromAPI(
                `${allUsersWithRolesEndpoint}?${params}`
            );
            // console.log(response);
            if (response) {
                const prepareForSelectbox = response.map((user: any) => {
                    return {
                        label: user.email,
                        value: user.email,
                    };
                });
                yield actions.setUsersEmailList(prepareForSelectbox);
                return prepareForSelectbox;
            }
        } catch (error) {
            console.error('Error fetching Users email', error);
        }
    },
};

export default actions;
