/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */
import {
    ISettingsAttributes,
    IsettingsMetaForm,
    settingsMetaDefaultData,
} from './state';
import * as Types from '../types';
import {
    jobAttributeMetasEndpoint,
    jobAttributeRestoreEndpoint,
    jobAttributeDeleteEndpoint,
} from '../endpoint';
import { IResponseGenerator } from '../../interfaces';
import { SwalError, SwalSuccess } from '../../utils/Swal';
import { formatSelect2Data } from '../../utils/Select2Helper';
import { sortAlphabetically } from '../../utils/Global';

const actions = {
    setSettingsAttribute(settingsAttributes: ISettingsAttributes) {
        return {
            type: Types.GET_SETTINGS_ATTRIBUTES,
            settingsAttributes,
        };
    },
    setSettingAttributeMetas(settingsAttributeMetas: ISettingsAttributes) {
        return {
            type: Types.GET_SETTINGS_ATTRIBUTE_METAS,
            settingsAttributeMetas,
        };
    },
    setSettingsAttributeMetaTotal(settingsAttributeMetaTotal: number) {
        return {
            type: Types.SET_SETTINGS_ATTRIBUTE_META_TOTAL,
            settingsAttributeMetaTotal,
        };
    },
    setSettingsAttributeMetaTotalPage(settingsAttributeMetaTotalPage: number) {
        return {
            type: Types.SET_SETTINGS_ATTRIBUTE_META_TOTAL_PAGE,
            settingsAttributeMetaTotalPage,
        };
    },
    setLoadingSettings(loadingSettings: boolean) {
        return {
            type: Types.SET_LOADING_SETTINGS,
            loadingSettings,
        };
    },
    setLoadingSettingsMeta(loadingSettingsMeta: boolean) {
        return {
            type: Types.SET_LOADING_SETTINGS_META,
            loadingSettingsMeta,
        };
    },
    setMetaFormData(settingsMetaForm: IsettingsMetaForm) {
        return {
            type: Types.SET_META_FORM_DATA,
            settingsMetaForm,
        };
    },
    setJobSettings(jobSettings: ISettingsAttributes) {
        return {
            type: Types.GET_JOB_SETTINGS,
            jobSettings,
        };
    },
    fetchFromAPI(path: string) {
        return {
            type: Types.FETCH_FROM_API,
            path,
        };
    },
    fetchFromAPIUnparsed(path: string) {
        return {
            type: Types.FETCH_FROM_API_UNPARSED,
            path,
        };
    },
    *getSettingAttributeMetas(filters: any) {
        if (filters === undefined) {
            filters = {};
        }
        const params = new URLSearchParams(
            filters as URLSearchParams
        ).toString();

        try {
            yield actions.setLoadingSettingsMeta(true);
            const path = `${jobAttributeMetasEndpoint}?${params}`;
            const response = yield actions.fetchFromAPIUnparsed(path);
            if (response.data) {
                const metaAttributeId = response?.data?.data[0]?.attribute_id;
                const metaAttributeLabel =
                    response?.data?.data[0]?.attribute_slug?.name;
                const responsData = {
                    ...response?.data,
                    metaAttributeId,
                    metaAttributeLabel,
                };

                let totalCount = 0;
                let totalPage = 0;
                if (response.headers !== undefined) {
                    totalPage = response.headers.get('X-WP-TotalPages');
                    totalCount = response.headers.get('X-WP-Total');
                }
                yield actions.setSettingsAttributeMetaTotal(totalCount);
                yield actions.setSettingsAttributeMetaTotalPage(totalPage);
                yield actions.setSettingAttributeMetas(responsData);
                yield actions.setLoadingSettingsMeta(false);
                return responsData;
            }
        } catch (error: any) {
            if (error.data && error.data.status === 403) {
                //  console.log('error from actions', error)
                SwalError({
                    title: __('Errors', 'ai-job-listing'),
                    text: error.message + ' - ' + error.data.status,
                    toast: true,
                    timer: 7000,
                    position: 'center',
                });
            } else {
                // For other errors, log the error
                console.error('Error fetching setting attribute metas:', error);
            }
        }
    },
    *setSettingAttributeSingleMeta(payload: number) {
        if (payload) {
            try {
                const path = `${jobAttributeMetasEndpoint}/${payload}`;
                const response = yield actions.fetchFromAPI(path);
                if (response) {
                    const { id, value, attribute_id } = response.data[0];
                    yield actions.setMetaFormData({
                        id,
                        value,
                        attribute_id,
                    });
                }
            } catch (error) {
                // eslint-disable-next-line no-console
                console.error(
                    'Error fetching setting attribute single metas:',
                    error
                );
            }
        }
    },
    *saveAttributeMeta(payload: IsettingsMetaForm) {
        yield actions.setLoadingSettingsMeta(true);

        try {
            let response: IResponseGenerator = {};
            if (payload.id) {
                response = yield {
                    type: Types.UPDATE_SETTINGS_ATTRIBUTE_META,
                    payload,
                    path: `${jobAttributeMetasEndpoint}/${payload.id}`,
                };
            } else {
                response = yield {
                    type: Types.CREATE_SETTINGS_ATTRIBUTE_META,
                    payload,
                    path: jobAttributeMetasEndpoint,
                };
            }
            if (!payload?.id && response.status !== 409) {
                SwalSuccess({
                    title: __('Done', 'ai-job-listing'),
                    text: 'Job Settings Created Successfuly',
                });
            } else if (payload?.id && response.status !== 409) {
                SwalSuccess({
                    title: __('Updated', 'ai-job-listing'),
                    text: 'Job Settings Updated Successfuly',
                });
            } else if (!payload?.id && response.status === 409) {
                SwalError({
                    title: __('Unsuccessful', 'ai-job-listing'),
                    text: response.message,
                    timer: 3000,
                });
            }
            yield actions.setMetaFormData({ ...settingsMetaDefaultData });
            yield actions.setLoadingSettingsMeta(false);
            return response;
        } catch (error: any) {
            yield actions.setMetaFormData({ ...settingsMetaDefaultData });
            yield actions.setLoadingSettingsMeta(false);
            SwalError({
                title: __('Errors', 'ai-job-listing'),
                text: error.message,
                toast: true,
                position: 'bottom',
            });
        }
    },
    *deleteSettingsMeta(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.DELETE_SETTINGS_META,
                payload,
                path: jobAttributeMetasEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Moved', 'ai-job-listing'),
                    text: __(
                        'Job Attribute Moved to Trash Successfully',
                        'ai-job-listing'
                    ),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Move to Trash Attribute Meta', error);
        }
    },
    *deletePermanentlySettingsMeta(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.DELETE_SETTINGS_META,
                payload,
                path: jobAttributeDeleteEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Deleted', 'ai-job-listing'),
                    text: __(
                        'Job Attribute Deleted Successfully',
                        'ai-job-listing'
                    ),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Delete Attribute Meta', error);
        }
    },
    *restoreSettingsMeta(payload: Array<number>) {
        try {
            const response: IResponseGenerator = yield {
                type: Types.UPDATE_SETTINGS_ATTRIBUTE_META,
                payload,
                path: jobAttributeRestoreEndpoint,
            };
            if (response) {
                SwalSuccess({
                    title: __('Restored', 'ai-job-listing'),
                    text: __(
                        'Job Attribute Restored Successfully',
                        'ai-job-listing'
                    ),
                    position: 'center',
                });
            }
            return response;
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Restore Attribute Meta', error);
        }
    },
    *getJobSettings() {
        const queryParams = {
            status: 'active',
            posts_per_page: '-1',
        };

        const params = new URLSearchParams(queryParams);
        try {
            const path = `${jobAttributeMetasEndpoint}?${params}`;
            const response = yield actions.fetchFromAPI(path);
            if (response) {
                const departments = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-departments'
                );
                const jobType = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-types'
                );
                const shifts = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-shifts'
                );
                const locations = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-locations'
                );
                const qualification = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-education-qualifications'
                );
                const jobMethod = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-application-method'
                );
                const salaryType = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'job-salary-type'
                );
                const experinceYears = response?.data?.filter(
                    (setting: any) =>
                        setting.attribute_slug?.slug == 'work-experience'
                );

                const settings: any = {
                    departments: formatSelect2Data(departments),
                    jobType: formatSelect2Data(jobType),
                    shifts: formatSelect2Data(shifts),
                    locations: formatSelect2Data(locations),
                    qualification: formatSelect2Data(qualification),
                    jobMethod: formatSelect2Data(jobMethod),
                    salaryType: formatSelect2Data(salaryType),
                    experinceYears: formatSelect2Data(experinceYears),
                };
                sortAlphabetically(settings?.departments, 'label');
                sortAlphabetically(settings?.jobType, 'label');
                sortAlphabetically(settings?.shifts, 'label');
                sortAlphabetically(settings?.locations, 'label');
                sortAlphabetically(settings?.qualification, 'label');
                sortAlphabetically(settings?.jobMethod, 'label');
                sortAlphabetically(settings?.salaryType, 'label');
                sortAlphabetically(settings?.experinceYears, 'label');

                yield actions.setJobSettings(settings);
            }
        } catch (error) {
            // eslint-disable-next-line no-console
            console.error('Error Fetching Settings:', error);
        }
    },
};

export default actions;
