/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useLocation, Link } from 'react-router-dom';
import ReactPlayer from 'react-player';

/**
 * Internal dependencies.
 */
import jobstore from '../../data/job';
import generalSettings from '../../data/general-settings';
import Button from '../../components/button/Button';
import LoadingBox from '../../components/loader/Loading';
import {
    formatSalaryAmount,
    getApplicantValueByKey,
    getValueByKey
} from '../../utils/Global';
import { formatDateOnly } from '../../utils/DateHelper';
import Table from '../../components/table/Table';
import pro_thumb from '../../style/images/pro_update.png';
import intro_thumb from '../../style/images/intro.png';
import deadline from '../../style/images/application-deadline.png';
import resume from '../../style/images/resume-preview.png';
import aidescription from '../../style/images/ai-description.png';
import emailNotification from '../../style/images/email-notification.png';
import searchFilter from '../../style/images/search-filter.png';
import applicationStatus from '../../style/images/application-status.png';
import LoaderLine from '../../components/loader/LoaderLine';

const AiJobDashboard = () => {
    const dispatch = useDispatch();
    const location = useLocation();
    const [queryParams, setQueryParams] = useState<any>({
        column: 'specific',
        status: ['active', 'inactive', 'expired', 'archived', 'draft'],
    });
    const [jobApplicationList, setJobApplicationList] = useState<Array<any>>([]);
    const [jobList, setJobList] = useState<Array<any>>([]);

    const jobOverview: boolean = useSelect(
        (select) => select(jobstore).getJobOverviewStatus(),
        []
    );
    const jobOverviewLoading: boolean = useSelect(
        (select) => select(jobstore).getJobOverviewStatusLoading(),
        []
    );
    
    const jobLoading: boolean = useSelect(
        (select) => select(jobstore).getJobListsLoading(),
        []
    );

    useEffect(() => {
        dispatch(jobstore).getJobLists(queryParams).
        then((response: any) => {
            if (response) {
                const sliceJob = response?.slice(0, 3);
                setJobList(sliceJob);
            }
        });
        dispatch(jobstore).setJobApplications().
        then((response: any) => {
            if (response) {
                const sliceApplication = response?.slice(0, 3);
                setJobApplicationList(sliceApplication);
            }
        });
       
    }, []);

    // Job List Table
    const jobTableHeader = [
        {
            key: 'title',
            title: __('Jobs', 'ai-job-listing'),
        },
        {
            key: 'total_applicants', 
            title: __('Applicants', 'ai-job-listing'),
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'job_departments',
            title: __('Department', 'ai-job-listing'),
        },
        {
            key: 'job_locations',
            title: __('Location', 'ai-job-listing'),
        },
        {
            key: 'application_deadline',
            title: __('Deadline', 'ai-job-listing')
        },
    ];

    // Job Applicant List
    const jobApplicationLoading: boolean = useSelect(
        (select) => select(jobstore).getJobApplicationLoading(),
        []
    );
   
    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    const jobLimitStatus: boolean = useSelect(
        (select) => select(jobstore).getHasJobLimit(),
        []
    );

    // console.log(jobLimitStatus);

    const applicantTableHeader = [
        {
            key: 'name',
            title: __('Applicant Name', 'ai-job-listing'),
        },
        {
            key: 'experience',
            title:  __('Experience', 'ai-job-listing'),
        },
        {
            key: 'applied_for',
            title:  __('Applied For', 'ai-job-listing'),
        },
        {
            key: 'expected_salary',
            title:  __('Expected Salary', 'ai-job-listing'),
            class: 'text-right',
            bodyClass: 'text-right',
        }
    ];

    // Sidebar Data
    const proFeatures = [
        {
            title: 'Auto Resume Suggest',
            icon: 'tio-checkmark_square_outlined',
            isAvailable: false,
        },
        {
            title: 'Job Alerts',
            icon: 'tio-checkmark_square_outlined',
            isAvailable: false
        },
        {
            title: 'Auto Delete Applications',
            icon: 'tio-checkmark_square_outlined',
            isAvailable: false,
        },
        {
            title: 'Automatic Job Matching AI',
            icon: 'tio-checkmark_square_outlined',
            isAvailable: false,
        }
    ]
    const quickLink = [
        {
            title: 'VIP Priority Support',
            icon: 'tio-headphones',
            link: '#'
        },
        {
            title: 'Help Center',
            icon: 'tio-info_outined',
            link: '#'
        }
    ]
    const proAddons = [
        {
            icon: deadline,
            title: 'Application Deadline',
            description: 'Let applicants know the exact deadlines up front.',
            isFree: true,
            isActive: true
        },
        {
            icon: resume,
            title: 'Quick Resume Preview',
            description: 'Quickly skim through applicants by key details.',
            isFree: true,
            isActive: false
        },
        {
            icon: aidescription,
            title: 'AI Job Description',
            description: 'Write clear, tailored job descriptions with AI',
            isFree: true,
            isActive: false
        },
        {
            icon: emailNotification,
            title: 'Email Notification',
            description: 'No candidate goes unnoticed, with real time alerts.',
            isFree: true,
            isActive: false
        },
        {
            icon: searchFilter,
            title: 'Search and Filter',
            description: 'Set your criteria to dig through applications.',
            isFree: true,
            isActive: false
        },
        {
            icon: applicationStatus,
            title: 'Application Status Manager',
            description: 'Easily sort candidates by stage.',
            isFree: true,
            isActive: false
        }
    ]

    return (
        <>
            <div className="w-100 flex gap-5">
                <div className="space-y-5 flex-1">
                    {/* Welcome Banner Start */}
                    <div className="p-[30px] rounded bg-white">
                        <div className="flex gap-5 xxl:gap-14">
                            <div className="flex-1">
                                <h2 className="mb-2.5">Welcome to AI Job Listing</h2>
                                <p>A secure, customizable AI-powered WordPress plugin that helps you manage jobs, applicants, and team workflows with ease. Built for growing businesses, HR professionals, and recruitment agencies looking to simplify hiring with zero complexities.</p>
                                {jobLimitStatus && <Button
                                    text={__('Create A Job', 'ai-job-listing')}
                                    href="/job/create"
                                    btnStyle="secondary"
                                    size="large"
                                />}
                            </div>
                            <div className="relative hidden m-auto xl:block">
                                <div className="w-full max-w-[340px] min-h-[190px] aspect-video rounded overflow-hidden">
                                    <ReactPlayer
                                        src="https://www.youtube.com/watch?v=J8tVn6HFFkM"
                                        light={<img src={intro_thumb} alt="intro" />}
                                        playIcon={<i className="hidden"></i>}
                                        controls={true}
                                        style={{ width: '100%', height: 'auto', aspectRatio: '16/9', borderRadius: '10px' }}
                                    />
                                </div>
                            </div>
                        </div>
                    </div>
                    {/* Welcome Banner End */}
                    {/* Overview Info Start */}
                    <div className="p-[30px] bg-white rounded">
                        <div className="grid grid-cols-3 gap-4">
                            {jobOverviewLoading ? (
                                <LoaderLine line={4} />
                                ) : (
                                <div className="p-[20px] xxl:p-[30px] xxl:flex-row flex flex-col-reverse items-center justify-between gap-5 border border-border-light rounded-md">
                                <>
                                    <div className="leading-[1] flex-1 text-center xxl:text-left">
                                        <div className="mb-3 text-[20px] text-headline">Total Job Posts</div>
                                        <div className="text-[32px] font-semibold text-[#2F3448]">
                                            {jobOverview?.total_jobs}
                                        </div>
                                    </div>
                                    <div className="flex items-center justify-center h-[67px] w-[67px] rounded-full text-[32px] bg-[#E3E3FF]">
                                        <i className="tio-briefcase text-purple"></i>
                                    </div>
                                </>
                            </div>
                            )}
                            {jobOverviewLoading ? (
                                <LoaderLine line={4} />
                                ) : (
                                <div className="p-[20px] xxl:p-[30px] xxl:flex-row flex flex-col-reverse items-center justify-between gap-5 border border-border-light rounded-md">
                                    <div className="leading-[1] flex-1 text-center xxl:text-left">
                                        <div className="mb-3 text-[20px] text-headline">Total Active Jobs</div>
                                        <div className="text-[32px] font-semibold text-[#2F3448]">{jobOverview?.total_active_jobs}</div>
                                    </div>
                                    <div className="flex items-center justify-center h-[67px] w-[67px] rounded-full text-[32px] bg-[#FFF2C5]">
                                        <i className="tio-file_text text-yellow"></i>
                                    </div>
                                </div>
                            )}
                            {jobOverviewLoading ? (
                                <LoaderLine line={4} />
                                ) : (
                            <div className="p-[20px] xxl:p-[30px] xxl:flex-row flex flex-col-reverse items-center justify-between gap-5 border border-border-light rounded-md">
                                <div className="leading-[1] flex-1 text-center xxl:text-left">
                                    <div className="mb-3 text-[20px] text-headline">Total Applicants</div>
                                    <div className="text-[32px] font-semibold text-[#2F3448]">{jobOverview?.total_applicants}</div>
                                </div>
                                <div className="flex items-center justify-center h-[67px] w-[67px] rounded-full text-[32px] bg-[#D6F3FF]">
                                    <i className="tio-group_senior text-info"></i>
                                </div>
                            </div>
                            )}
                        </div>
                    </div>
                    {/* Overview Info End */}
                    {/* Recent Jobs Table Start */}
                    <div className="p-[30px] rounded bg-white dashboard-table">
                        <div className="flex items-center justify-between gap-5">
                            <h3 className="text-primary">Recent Job Post</h3>
                            <div className="mb-5">
                                <Link to="/job-list" className="flex items-center gap-1.5 text-success text-b2-regular ring-0 focus:ring-0 transition-all hover:text-primary">
                                    View All <i className="tio-arrow_forward"></i>
                                </Link>
                            </div>
                        </div>
                        {jobLoading ? (
                            <LoadingBox block={jobList.length} />
                            ) : (
                            <Table header={jobTableHeader} body={jobList}>
                                <div customKey="title">
                                    {(data: any) => (
                                        <>
                                            <h5 className="mb-[5px]">
                                                {data.title}
                                            </h5>
                                        </>
                                    )}
                                </div>
                                <div customKey="job_departments">
                                    {(data: any) => (
                                        getValueByKey(
                                            data?.attribute_values,
                                            'job_departments'
                                        ) && (
                                            <span>
                                                {getValueByKey(
                                                    data?.attribute_values,
                                                    'job_departments'
                                                )}
                                            </span>
                                        )
                                    )}
                                </div>
                                <div customKey="job_locations">
                                    {(data: any) => (
                                        getValueByKey(
                                            data?.attribute_values,
                                            'job_locations'
                                        ) && (
                                            <span>
                                                {getValueByKey(
                                                    data?.attribute_values,
                                                    'job_locations'
                                                )}
                                            </span>
                                        )
                                    )}
                                </div>
                                <div customKey="application_deadline">
                                    {(data: any) => (
                                        formatDateOnly(
                                            data?.application_deadline
                                        ) && (
                                            <span>
                                                {formatDateOnly(
                                                    data?.application_deadline
                                                )}
                                            </span>
                                        )
                                    )}
                                </div>
                            </Table>
                        )}
                    </div>
                    {/* Recent Jobs Table End */}
                    {/* Recent Application Table Start */}
                    <div className="p-[30px] rounded bg-white dashboard-table">
                        <div className="flex items-center justify-between gap-5">
                            <h3 className="text-primary">Recent Applicant</h3>
                            <div className="mb-5">
                                <Link to="/applicants" className="flex items-center gap-1.5 text-success text-b2-regular ring-0 focus:right-0 transition-all hover:text-primary">
                                    View All <i className="tio-arrow_forward"></i>
                                </Link>
                            </div>
                        </div>
                        <>
                        {jobApplicationLoading ? (
                            <LoadingBox block={jobApplicationList?.length} />
                        ) : (
                            <Table
                                header={applicantTableHeader}
                                body={jobApplicationList}
                            >
                                <div customKey="name">
                                    {(applicant: any) => (
                                        <div className="flex items-center leading-[1.2]">
                                            {applicant.profile_image_url ? (
                                                <img
                                                    className="w-8 h-8 m-0 rounded-full"
                                                    src={applicant.profile_image_url}
                                                    alt={getApplicantValueByKey(
                                                        applicant?.meta_attributes,
                                                        'full_name'
                                                    )}
                                                />
                                            ) : (
                                                <span className="rounded-full text-placeholder text-[32px]">
                                                    <i className="tio-account_circle"></i>
                                                </span>
                                            )}
                                            <span className="ml-2.5 mb-0 font-medium text-headline">
                                                {getApplicantValueByKey(
                                                    applicant?.meta_attributes,
                                                    'full_name'
                                                )}
                                            </span>
                                        </div>
                                    )}
                                </div>
                                <div customKey="experience">
                                    {(applicant: any) =>
                                        getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'work_experience'
                                        ) && (
                                            <>
                                                {getApplicantValueByKey(
                                                    applicant?.meta_attributes,
                                                    'work_experience'
                                                )}
                                            </>
                                        )
                                    }
                                </div>
                                <div customKey="applied_for">
                                    {(applicant: any) => (
                                        <span>
                                            {applicant?.job_title?.title}
                                        </span>
                                    )}
                                </div>
                                <div customKey="expected_salary">
                                    {(applicant: any) =>
                                        getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'expected_salary'
                                        ) && (
                                            <>
                                                {settings?.currency_symbol}
                                                {formatSalaryAmount(
                                                    getApplicantValueByKey(
                                                        applicant?.meta_attributes,
                                                        'expected_salary'
                                                    )
                                                )}
                                            </>
                                        )
                                    }
                                </div>
                            </Table>
                        )}
                    </>
                    </div>
                    {/* Recent Application Table End */}
                </div>
                {/* Sidebar Area Start */}
                <div className="w-[450px] xxl:w-[540px] space-y-5">
                    {/* Upgrade Pro Notice Start */}
                    <div className="px-[30px] pt-[30px] pb-[15px] rounded bg-white text-center">
                        <div className="m-5">
                            <img
                                className="my-0 mx-auto"
                                src={pro_thumb}
                                alt="pro update"
                            />
                        </div>
                        <h2 className="text-h1 mb-3">Get More Done With Premium</h2>
                        <p>Further automate your job board and reduce busy work with the premium features of AI Job Listing. </p>
                        {/* <Button
                            text={__('Upgrade Pro', 'ai-job-listing')}
                            href="#"
                            btnStyle="primary"
                            size="large"
                        /> */}
                        <div className="mt-3.5">
                            {proFeatures.map((data, index) => (
                                <div key={index} className={`py-3.5 flex items-center border-border-light ${index === proFeatures.length - 1 ? '' : 'border-b'}`}>
                                    <i className={`${data.icon} text-primary text-[20px]`}></i>
                                    <div className="pl-5 text-b3-mid text-headline">{data.title}</div>
                                    <div className="pl-2 text-b3-mid text-secondary">{data.isAvailable ? "" : "(Coming Soon)"}</div>
                                </div>
                            ))}
                        </div>
                    </div>
                    {/* Upgrade Pro Notice End */}
                    {/* Pro Addon Start */}
                    <div className="p-[30px] rounded bg-white">
                        <h3 className="text-primary">AI Job Listing Features</h3>
                        <div className="grid grid-cols-2 gap-2.5">
                            {proAddons.map((data, index) => (
                                <div className="p-5 border border-border-light rounded-md" key={index}>
                                    <div className="mb-5 flex items-center justify-between gap-2">
                                        <div>
                                            <img
                                                className="my-0 mx-auto"
                                                src={data.icon}
                                                alt={data.title}
                                            />
                                        </div>
                                        <div className="flex items-center gap-2.5">
                                            <div className={`px-2.5 py-[1px] border rounded-sm  text-[13px] font-bold leading-[16px] ${data.isFree ? 'border-green text-green bg-success-light' : 'border-info text-info bg-info-light'}`}>{data.isFree ? 'Free' : 'PRO'}</div>
                                        </div>
                                    </div>
                                    <div>
                                        <h6 className="mb-1.5">{data.title}</h6>
                                        <p className="m-0 text-[14px] leading-[20px]">{data.description}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                    {/* Pro Addon End */}
                    {/* Quick Access Link Start */}
                    {/* <div className="px-[30px] pt-[30px] pb-[15px] rounded bg-white">
                        <h2 className="mb-0 text-primary">Quick Access</h2>
                        <div>
                            {quickLink.map((data, index) => (
                                <a key={index} href={data.link} className={`py-4 flex items-center gap-5 border-border-light focus:ring-0 ${index === quickLink.length - 1 ? '' : 'border-b'}`} target="_blank" rel="noopener noreferrer">
                                    <i className={`${data.icon} text-primary text-[18px]`}></i>
                                    <div className="text-b3-mid text-headline">{data.title}</div>
                                </a>
                            ))}
                        </div>
                    </div> */}
                    {/* Quick Access Link End */}
                </div>
                {/* Sidebar Area End */}
            </div>
        </>
    );
};

export default AiJobDashboard;
