/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { Menu } from '@headlessui/react';
const lodash = require('lodash');
/**
 * Internal dependencies.
 */
import jobstore from '../../../data/job';
import generalSettings from '../../../data/general-settings';
import PageTitle from '../../../components/layout/PageTitle';
import FilterBar from '../../../components/table/FilterBar';
import LoadingBox from '../../../components/loader/Loading';
import {
    formatSalaryAmount,
    getApplicantValueByKey,
} from '../../../utils/Global';
import Pagination from '../../../components/pagination/Pagination';
import SearchInput from '../../../components/form/SearchInput';
import Table from '../../../components/table/Table';
import Dropdown from '../../../components/dropdown/Dropdown';
import ApplicationPreviewModal from '../../../components/modal/ApplicationPreviewModal';
import { Link } from 'react-router-dom';
import { SwalWarning } from '../../../utils/Swal';
import Button from '../../../components/button/Button';

const ApplicantList = () => {
    const dispatch = useDispatch();
    const [queryParams, setQueryParams] = useState<any>({});

    const jobApplicationList: Array<any> = useSelect(
        (select) => select(jobstore).getJobApplicationList(queryParams),
        []
    );
    const jobApplicationCount: boolean = useSelect(
        (select) => select(jobstore).getJobApplicationTotalCount(),
        []
    );
    const jobApplicationLoading: boolean = useSelect(
        (select) => select(jobstore).getJobApplicationLoading(),
        []
    );

    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    const getUser: object = useSelect(
        (select) => select(generalSettings).getCurrentUser(),
        []
    );

    useEffect(() => {
        dispatch(jobstore).setJobApplications(queryParams);
    }, [queryParams]);

    const applicantTableHeader = [
        {
            key: 'name',
            title: __('Name', 'ai-job-listing'),
        },
        {
            key: 'experience',
            title:  __('Experience', 'ai-job-listing'),
        },
        {
            key: 'available_start',
            title:  __('Available Start', 'ai-job-listing'),
        },
        {
            key: 'applied_for',
            title:  __('Applied For', 'ai-job-listing'),
        },
        {
            key: 'expected_salary',
            title:  __('Expected Salary', 'ai-job-listing'),
            class: 'text-right',
            bodyClass: 'text-right',
        },
        {
            key: 'status',
            title: 'Status',
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'actions',
            title: '',
            bodyClass: 'text-right',
        },
    ];

    const applicantPaginationHandler = (pages: any) => {
        if (pages != null) {
            dispatch(jobstore).setJobApplications({
                ...queryParams,
                page: pages,
            });
        }
    };

    const applicantSearchHandler = lodash.debounce((text: string) => {
        setQueryParams({ ...queryParams, search: text, page: 1 });
    }, 500);

    // Applicant View Details Handler
    const [isPreviewModalOpen, setIsPreviewModalOpen] = useState(false);
    const [jobSingleApplication, setJobSingleApplication] = useState<any>({});
    const [applicationComments, setApplicationComments] = useState<Array<any>>(
        []
    );
    const [commentText, setCommentText] = useState<string>('');
    const [commentFormError, setCommentFormError] = useState<string>('');

    function closePreviewModal() {
        setIsPreviewModalOpen(false);
        setCommentText('');
    }

    const applicantPreviewHandler = (applicationId: string) => {
        setIsPreviewModalOpen(true);

        // Get Applicant Details
        dispatch(jobstore)
            .setJobApplicationDetails({ id: applicationId })
            .then((response) => {
                setJobSingleApplication(response);
            });

        // Get Comments
        dispatch(jobstore)
            .setApplicationCommentById({
                app_id: applicationId,
            })
            .then((res: any) => {
                if (res) {
                    setApplicationComments(res);
                }
            })
            .catch((err: any) => {
                console.error(err);
            });
    };

    const submitCommentHandler = (e: any) => {
        e.preventDefault();
        const prepareComment = {
            user_id: getUser.id || null,
            app_id: jobSingleApplication.id || null,
            app_process_id: jobSingleApplication.process_id || null,
            comment: commentText,
        };
        if (commentText.length < 1) {
            setCommentFormError(__('Please enter a character', 'ai-job-listing'));
        } else {
            setCommentFormError('');
            dispatch(jobstore)
                .saveApplicationComment(prepareComment)
                .then((res: any) => {
                    if (res) {
                        setApplicationComments([res, ...applicationComments]);
                        setCommentText('');
                    }
                })
                .catch((err: any) => {
                    // eslint-disable-next-line no-console
                    console.error(err);
                });
        }
    };

    const applicantDeleteHandler = (id: any) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure move to trash?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(jobstore)
                    .deleteJobApplication({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.message) {
                            setQueryParams({ ...queryParams });
                        }
                    });
            },
        });
    };

    return (
        <>
            <PageTitle text={__('All Applicants', 'ai-job-listing')} />
            <FilterBar
                leftItem={
                    <SearchInput
                        name="applicant_search"
                        placeholder={__('Search', 'ai-job-listing')}
                        onChange={(text) => {
                            applicantSearchHandler(text);
                        }}
                    />
                }
            ></FilterBar>
            {/* Application List Table */}
            <>
                {jobApplicationLoading ? (
                    <LoadingBox block={jobApplicationList?.length} />
                ) : (
                    <Table
                        header={applicantTableHeader}
                        body={jobApplicationList}
                    >
                        <div customKey="name">
                            {(applicant: any) => (
                                <button
                                    type="button"
                                    className="flex items-center leading-[1.2]"
                                    onClick={() =>
                                        applicantPreviewHandler(applicant.id)
                                    }
                                >
                                    {applicant.profile_image_url ? (
                                        <img
                                            className="w-8 h-8 m-0 rounded-full"
                                            src={applicant.profile_image_url}
                                            alt={getApplicantValueByKey(
                                                applicant?.meta_attributes,
                                                'full_name'
                                            )}
                                        />
                                    ) : (
                                        <span className="rounded-full text-placeholder text-[32px]">
                                            <i className="tio-account_circle"></i>
                                        </span>
                                    )}
                                    <span className="ml-2.5 mb-0">
                                        {getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'full_name'
                                        )}
                                    </span>
                                </button>
                            )}
                        </div>
                        <div customKey="experience">
                            {(applicant: any) =>
                                getApplicantValueByKey(
                                    applicant?.meta_attributes,
                                    'work_experience'
                                ) && (
                                    <>
                                        {getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'work_experience'
                                        )}
                                    </>
                                )
                            }
                        </div>
                        <div customKey="available_start">
                            {(applicant: any) =>
                                getApplicantValueByKey(
                                    applicant?.meta_attributes,
                                    'when_join'
                                ) && (
                                    <>
                                        {getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'when_join'
                                        )}
                                    </>
                                )
                            }
                        </div>
                        <div customKey="applied_for">
                            {(applicant: any) => (
                                <Link
                                    to={`/job/list/${applicant?.job_id}`}
                                    className="text-b2-regular text-primary underline underline-offset-[3px]"
                                >
                                    {applicant?.job_title?.title}
                                </Link>
                            )}
                        </div>
                        <div customKey="expected_salary">
                            {(applicant: any) =>
                                getApplicantValueByKey(
                                    applicant?.meta_attributes,
                                    'expected_salary'
                                ) && (
                                    <>
                                        {settings?.currency_symbol}
                                        {formatSalaryAmount(
                                            getApplicantValueByKey(
                                                applicant?.meta_attributes,
                                                'expected_salary'
                                            )
                                        )}
                                    </>
                                )
                            }
                        </div>
                        <div customKey="status">
                            {(applicant: any) => (
                                <span
                                    className="inline-flex items-center gap-[5px] w-[110px] px-[10px] pt-1 pb-[3px] rounded-xsm text-white text-[15px]"
                                    style={{
                                        backgroundColor:
                                            applicant?.app_process?.icon_color,
                                    }}
                                >
                                    <i
                                        className={`${applicant?.app_process?.icon} text-[14px]`}
                                    ></i>
                                    {applicant?.app_process?.name}
                                </span>
                            )}
                        </div>
                        <div customKey="actions">
                            {(applicant: any) => (
                                <Dropdown>
                                    <Menu.Item>
                                        <Button
                                            text={__('Delete', 'ai-job-listing')}
                                            btnStyle="transparentSecondary"
                                            outline
                                            onClick={() =>
                                                applicantDeleteHandler(
                                                    applicant.id
                                                )
                                            }
                                        />
                                    </Menu.Item>
                                </Dropdown>
                            )}
                        </div>
                    </Table>
                )}
            </>
            <Pagination
                perPage={settings?.posts_per_page}
                total={jobApplicationCount}
                paginate={(pages: number) => {
                    applicantPaginationHandler(pages);
                }}
            />
            {/* Applicant Details Modal Start */}
            <ApplicationPreviewModal
                isPreviewModalOpen={isPreviewModalOpen}
                closePreviewModal={closePreviewModal}
                jobSingleApplication={jobSingleApplication}
                commentText={commentText}
                commentFormError={commentFormError}
                setCommentText={(text) => setCommentText(text)}
                submitCommentHandler={submitCommentHandler}
                applicationComments={applicationComments}
            />
        </>
    );
};

export default ApplicantList;
