/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useLocation, Link } from 'react-router-dom';
const lodash = require('lodash');
import { Menu } from '@headlessui/react';

/**
 * Internal dependencies.
 */
import jobstore from '../../../data/job';
import generalSettings from '../../../data/general-settings';
import PageTitle from '../../../components/layout/PageTitle';
import FilterBar from '../../../components/table/FilterBar';
import Button from '../../../components/button/Button';
import { Input } from '../../../components/form/Input';
import { defaultJobPostForm } from '../../../data/job/state';
import LoadingBox from '../../../components/loader/Loading';
import { selectStylesSM } from '../../../components/form/StyleClass';
import {
    getValueByKey,
    mergeAndFilterDuplicateArraysByKey,
} from '../../../utils/Global';
import Pagination from '../../../components/pagination/Pagination';
import TabButton from '../../../components/button/TabButton';
import SearchInput from '../../../components/form/SearchInput';
import Table from '../../../components/table/Table';
import { formatDateOnly } from '../../../utils/DateHelper';
import { statusElementGenerate } from '../../../utils/Element';
import SwitchCheckbox from '../../../components/form/SwitchCheckbox';
import { SwalSuccess, SwalWarning } from '../../../utils/Swal';
import Dropdown from '../../../components/dropdown/Dropdown';

const JobsList = () => {
    const dispatch = useDispatch();
    const location = useLocation();
    const [trash, setTrash] = useState(false);
    const [jobStatusCount, setJobStatusCount] = useState<Array<any>>([]);
    const [currentStatusTabInfo, setCurrentStatusTabInfo] = useState<Object>(
        {}
    );
    const [queryParams, setQueryParams] = useState<any>({
        column: 'specific',
        status: ['active', 'inactive', 'expired', 'archived', 'draft'],
    });
    const [proLimit, setProLimit] = useState<boolean>(false);

    const getJobList: Array<any> = useSelect(
        (select) => select(jobstore).getJobLists(queryParams),
        []
    );
    const jobListCount: Array<any> = useSelect(
        (select) => select(jobstore).getJobListsTotalCount(),
        []
    );
    const getJobStatusCount: Array<any> = useSelect(
        (select) => select(jobstore).getJobStatusCount(),
        []
    );
    const jobLoading: boolean = useSelect(
        (select) => select(jobstore).getJobListsLoading(),
        []
    );

    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    // Fetch API when come from others route
    useEffect(() => {
        if (location.key != 'default') {
            dispatch(jobstore).getJobLists(queryParams);
            dispatch(jobstore).getJobStatusCount();
        }
        dispatch(jobstore)
            .getHasJobLimit()
            .then((response: any) => {
                setProLimit(response);
            });
    }, []);

    const jobTrashHandler = (select: any) => {
        setTrash(select.value.value);
        dispatch(jobstore).getJobLists(
            select.value.value
                ? {
                      column: 'specific',
                      trash: select.value.value,
                      status: queryParams.status,
                  }
                : {
                      column: 'specific',
                      status: queryParams.status,
                  }
        );
    };

    const jobListSearchHandler = lodash.debounce((text: string) => {
        dispatch(jobstore).getJobLists({
            ...queryParams,
            search: text,
            page: 1,
        });
        setQueryParams({
            ...queryParams,
            search: text,
        });
    }, 500);

    const jobListHandlerByStatus = (statusValue: any) => {
        dispatch(jobstore).getJobLists({
            column: 'specific',
            status: statusValue,
        });
        setQueryParams({
            column: 'specific',
            status: statusValue,
        });
        setTrash(false);
    };

    const jobPaginationHandler = (pages: any) => {
        if (pages != null) {
            dispatch(jobstore).getJobLists({
                ...queryParams,
                page: pages,
            });
        }
    };

    const jobStatusList = [
        {
            label: __('All Jobs', 'ai-job-listing'),
            value: ['active', 'inactive', 'expired', 'archived', 'draft'],
        },
        {
            label: __('Active', 'ai-job-listing'),
            value: ['active'],
        },
        {
            label: __('Expired', 'ai-job-listing'),
            value: ['expired'],
        },
        {
            label: __('Archived', 'ai-job-listing'),
            value: ['archived'],
        },
        {
            label: __('Paused', 'ai-job-listing'),
            value: ['inactive'],
        },
        {
            label: __('Draft', 'ai-job-listing'),
            value: ['draft'],
        },
    ];

    useEffect(() => {
        const mergeStatusAndCount = mergeAndFilterDuplicateArraysByKey(
            getJobStatusCount,
            jobStatusList,
            'label'
        );
        setJobStatusCount(mergeStatusAndCount);
    }, [getJobStatusCount]);

    // Job List Table
    const jobTableHeader = [
        {
            key: 'title',
            title: __('Jobs', 'ai-job-listing'),
        },
        {
            key: 'total_applicants', 
            title: __('Applicants', 'ai-job-listing'),
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'total_in_progress',
            title: __('In Progress', 'ai-job-listing'),
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'total_rejected',
            title: __('Rejected', 'ai-job-listing'),
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'total_hired',
            title: __('Hired', 'ai-job-listing'),
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'status',
            title: __('Status', 'ai-job-listing'),
            class: 'w-[200px] text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'actions',
            title: __('Actions', 'ai-job-listing'),
            class: 'w-[220px]',
        },
    ];
    const jobStatusUpdateHandler = (job: any) => {
        dispatch(jobstore)
            .updateJobField({
                id: Number(job.id),
                status: job.status,
            })
            .then((result: any) => {
                if (result.status) {
                    SwalSuccess({
                        title: __('Successfull', 'ai-job-listing'),
                        text: __(
                            'Successfully Updated the job status',
                            'ai-job-listing'
                        ),
                        showLoaderOnConfirm: true,
                        timer: 1500,
                    });
                    dispatch(jobstore).getJobLists({ ...queryParams });
                    dispatch(jobstore).getJobStatusCount();
                }
            });
    };
    const jobDeleteHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure to move trash?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(jobstore)
                    .deleteJobPost({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            dispatch(jobstore).getJobLists({ ...queryParams });
                            dispatch(jobstore).getJobStatusCount();
                        }
                    });
            },
        });
    };
    const jobPermanentDeleteHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are want to delete permanently?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(jobstore)
                    .deletePermanentJobPost({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            dispatch(jobstore).getJobLists({
                                ...queryParams,
                                trash: true,
                            });
                            dispatch(jobstore).getJobStatusCount();
                            dispatch(jobstore)
                                .getHasJobLimit()
                                .then((response: any) => {
                                    setProLimit(response);
                                });
                        }
                    });
            },
        });
    };
    const jobRestoreHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure to restore the job?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(jobstore)
                    .restoreTrashJob({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            dispatch(jobstore).getJobLists({ ...queryParams });
                            dispatch(jobstore).getJobStatusCount();
                            setTrash(false);
                        }
                    });
            },
        });
    };

    const user: any = useSelect((select) =>
        select(generalSettings).getCurrentUser()
    );

    return (
        <>
            <div className="flex items-start justify-between">
                <PageTitle text= {__('All Jobs', 'ai-job-listing')} />
                <TabButton
                    data={jobStatusCount}
                    onClick={(status) => {
                        jobListHandlerByStatus(status.value);
                        setCurrentStatusTabInfo(status);
                    }}
                    activeValue={queryParams.status}
                />
            </div>
            <FilterBar
                leftItem={
                    <>
                        <SearchInput
                            placeholder={__('Search jobs...', 'ai-job-listing')}
                            onChange={(text) => {
                                jobListSearchHandler(text);
                            }}
                        />
                    </>
                }
                rightItem={
                    <>
                        <Input
                            type="select"
                            name="job_trash"
                            value={trash}
                            select2Style={selectStylesSM}
                            options={[
                                { value: false, label: 'Active' },
                                { value: true, label: 'Trash' },
                            ]}
                            onChange={jobTrashHandler}
                        />
                        
                        <Button
                            text={__('Post New Job', 'ai-job-listing')}
                            href="/job/create"
                            btnStyle="secondary"
                            icon="tio-add text-[16px]"
                            disabled={proLimit ? false : true}
                            onClick={() => {
                                dispatch(jobstore).setJobPostForm({
                                    ...defaultJobPostForm,
                                });
                                dispatch(jobstore).setJobFormStep(1);
                                dispatch(jobstore).setJobFormStepCompleted(
                                    1
                                );
                            }}
                        />
                    </>
                }
            ></FilterBar>
            {jobLoading ? (
                <LoadingBox block={getJobList.length} />
            ) : (
                <Table header={jobTableHeader} body={getJobList}>
                    <div customKey="title">
                        {(data: any) => (
                            <>
                                <h5 className="mb-[5px]">
                                    <Link
                                        to={`/job/list/${data.id}`}
                                        className="decoration-transparent font-bold hover:text-primary focus:ring-0"
                                    >
                                        {data.title}
                                    </Link>
                                </h5>
                                {getValueByKey(
                                    data?.attribute_values,
                                    'job_departments'
                                ) && (
                                    <div className="mb-[15px] text-b2-table">
                                        <span>
                                            {getValueByKey(
                                                data?.attribute_values,
                                                'job_departments'
                                            )}
                                        </span>
                                    </div>
                                )}
                                <div className="flex items-center gap-2.5">
                                    {formatDateOnly(
                                        data?.application_deadline
                                    ) && (
                                        <div className="flex items-center gap-[5px] text-b1-notice-comment">
                                            <i className="tio-calendar"></i>
                                            <span>
                                                {formatDateOnly(
                                                    data?.application_deadline
                                                )}
                                            </span>
                                        </div>
                                    )}
                                    {data?.status &&
                                        statusElementGenerate(data.status)}
                                </div>
                            </>
                        )}
                    </div>
                    <div customKey="total_applicants">
                        {(data: any) => (
                            <div className="flex items-center justify-center">
                                {data?.total_applicants && (
                                    <>
                                        <span
                                            className={
                                                data?.total_unread > '0'
                                                    ? 'font-bold'
                                                    : ''
                                            }
                                        >
                                            {data?.total_unread}
                                        </span>{' '}
                                        /<span>{data?.total_applicants}</span>
                                    </>
                                )}
                            </div>
                        )}
                    </div>
                    <div customKey="status">
                        {(data: any) =>
                            data?.deleted_at === null &&
                            data?.status !== 'draft' &&
                            data?.status !== 'expired' && (
                                <SwitchCheckbox
                                    id="status"
                                    initialValue={
                                        // eslint-disable-next-line eqeqeq
                                        data.status == 'active' ? true : false
                                    }
                                    onChange={(val: any) =>
                                        jobStatusUpdateHandler({
                                            status:
                                                val?.value === true
                                                    ? 'active'
                                                    : 'inactive',
                                            id: data.id,
                                            currentStatus: data.status,
                                        })
                                    }
                                />
                            )
                        }
                    </div>
                    <div customKey="actions">
                        {(data: any) => (
                            <>
                                {data?.deleted_at === null && (
                                    <div className="flex items-center justify-end gap-2.5">
                                        <Button
                                            text={__('View Details', 'ai-job-listing')}
                                            btnStyle="action"
                                            outline
                                            href={`/job/list/${data.id}`}
                                        />
                                        <Button
                                            href={`/job/edit/${data.id}`}
                                            text={
                                                data?.status == 'archived'
                                                    ? __('Edit', 'ai-job-listing')
                                                    : __('Edit', 'ai-job-listing')
                                            }
                                            btnStyle="action"
                                            outline
                                            onClick={() => {
                                                dispatch(
                                                    jobstore
                                                ).setJobPostForm({
                                                    ...defaultJobPostForm,
                                                });
                                            }}
                                        />
                                        <Dropdown>
                                            <Menu.Item>
                                                <Button
                                                    href={`/job/edit/${data.id}`}
                                                    text={
                                                        data?.status ===
                                                        'archived'
                                                            ? __('Repost', 'ai-job-listing')
                                                            : __('Edit', 'ai-job-listing')
                                                    }
                                                    btnStyle="transparentPrimary"
                                                    outline
                                                />
                                            </Menu.Item>
                                            <Menu.Item>
                                                <Button
                                                    text={__('Delete', 'ai-job-listing')}
                                                    btnStyle="transparentSecondary"
                                                    outline
                                                    onClick={() =>
                                                        jobDeleteHandler(
                                                            data.id
                                                        )
                                                    }
                                                />
                                            </Menu.Item>
                                            {data?.status !== 'archived' &&
                                            data?.status !== 'draft' ? (
                                                <Menu.Item>
                                                    <Button
                                                        text={__('Archive', 'ai-job-listing')}
                                                        btnStyle="transparentPrimary"
                                                        outline
                                                        onClick={() =>
                                                            jobStatusUpdateHandler(
                                                                {
                                                                    status: 'archived',
                                                                    id: data.id,
                                                                    currentStatus:
                                                                        data.status,
                                                                }
                                                            )
                                                        }
                                                    />
                                                </Menu.Item>
                                            ) : null}
                                        </Dropdown>
                                    </div>
                                )}
                                {data?.deleted_at !== null && (
                                    <div className="flex items-center justify-end gap-2.5 w-[270px]">
                                        <Button
                                            text={__('Restore Job', 'ai-job-listing')}
                                            btnStyle="action"
                                            onClick={() =>
                                                jobRestoreHandler(data.id)
                                            }
                                            outline
                                        />
                                        <Button
                                            text={__('Delete Permanantly', 'ai-job-listing')}
                                            btnStyle="action"
                                            onClick={() =>
                                                jobPermanentDeleteHandler(
                                                    data.id
                                                )
                                            }
                                            outline
                                        />
                                    </div>
                                )}
                            </>
                        )}
                    </div>
                </Table>
            )}
            <Pagination
                perPage={settings?.posts_per_page}
                total={jobListCount}
                paginate={(pages: number) => {
                    jobPaginationHandler(pages);
                }}
            />
        </>
    );
};

export default JobsList;
