/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useParams } from 'react-router-dom';

/**
 * Internal dependencies.
 */
import jobstore from '../../../../data/job';
import generalSettings from '../../../../data/general-settings';
import JobApplications from './tab/JobApplications';
import JobSummery from './tab/JobSummery';
import JobReports from './tab/JobReports';
import { getValueByKey } from '../../../../utils/Global';
import { formatDateOnly } from '../../../../utils/DateHelper';
import PageTitle from '../../../../components/layout/PageTitle';
import { statusElementGenerate } from '../../../../utils/Element';
import TabButton from '../../../../components/button/TabButton';
import Error404 from '../../Error404';
import LoadingBox from '../../../../components/loader/Loading';

const JobDetails = () => {
    const params = useParams();
    const jobId = params.id;
    const dispatch = useDispatch();
    const [getJob, setGetJob] = useState<object>({});
    const [jobLoader, setJobLoader] = useState<boolean>(true);
    const [queryParams, setQueryParams] = useState<any>({
        column: '*',
        id: jobId,
        display_count: true,
    });

    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );
    const jobLoading: boolean = useSelect(
        (select) => select(jobstore).getJobDetailsLoading(),
        []
    );

    useEffect(() => {
        dispatch(jobstore)
            .setJobDetails(queryParams)
            .then((response: any) => {
                setJobLoader(jobLoading);
                setGetJob(response);
            });
    }, [queryParams]);

    const tabHeadingList = [
        {
            label: __('Applications', 'ai-job-listing'),
            value: 'applications',
            element: <JobApplications />,
        },
        {
            label: __('Report', 'ai-job-listing'),
            value: 'report',
            element: <JobReports getJobDetails={getJob} />,
        },
        {
            label: __('Job Preview', 'ai-job-listing'),
            value: 'job_preview',
            element: <JobSummery getJobDetails={getJob} settings={settings} />,
        },
    ];
    const [tabContent, setTabContent] = useState<any>(tabHeadingList[0]);

    return (
        <div
            className="job-application-list relative"
            style={{ minHeight: 'calc(100vh - 160px)' }}
        >
            {getJob?.id !== jobId ? (
                (jobLoader && <LoadingBox block={2} />) ||
                (!jobLoader && <Error404 />)
            ) : (
                <>
                    <div className="mb-[20px] flex items-start justify-between">
                        <div>
                            <PageTitle
                                text={getJob?.title}
                                areaClass="mb-[15px]"
                                backButton
                            />
                            <div className="flex items-center gap-2.5">
                                <div className="flex items-center gap-1 text-[16px] text-b2-table">
                                    <span className="flex items-center gap-[5px] text-headline">
                                        <i className="tio-briefcase_outlined"></i>
                                        Department:
                                    </span>
                                    <span>
                                        {getValueByKey(
                                            getJob?.attribute_values,
                                            'job_departments'
                                        )}
                                    </span>
                                </div>
                                <div className="flex items-center gap-1 text-[16px] text-b2-table">
                                    <span className="flex items-center gap-[5px] text-headline">
                                        <i className="tio-calendar"></i>
                                        Posted on:
                                    </span>
                                    <span>{formatDateOnly(getJob?.created_at)}</span>
                                </div>
                                <div className="flex items-center gap-1 text-[16px] text-b2-table">
                                    <span className="flex items-center gap-[5px] text-headline">
                                        <i className="tio-calendar"></i>
                                        Deadline:
                                    </span>
                                    <span>
                                        {formatDateOnly(getJob?.application_deadline)}
                                    </span>
                                </div>
                                {statusElementGenerate(getJob?.status)}
                            </div>
                        </div>
                        <TabButton
                            data={tabHeadingList}
                            onClick={(data) => setTabContent(data)}
                            activeValue={tabContent?.value}
                        />
                    </div>
                    {tabContent?.element}
                </>
            )}
        </div>
    );
};

export default JobDetails;
