/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useParams } from 'react-router-dom';
const lodash = require('lodash');

/**
 * Internal dependencies.
 */
import jobstore from '../../../../../data/job';
import generalSettings from '../../../../../data/general-settings';
import FilterBar from '../../../../../components/table/FilterBar';
import {
    getApplicantValueByKey,
    getApplicationDetailsWithStatusById,
} from '../../../../../utils/Global';
import Button from '../../../../../components/button/Button';
import LoadingBox from '../../../../../components/loader/Loading';
import { formatDateOnly } from '../../../../../utils/DateHelper';
import Pagination from '../../../../../components/pagination/Pagination';
import SearchInput from '../../../../../components/form/SearchInput';
import Table from '../../../../../components/table/Table';
import ApplicationCommentModal from '../../../../../components/modal/ApplicationCommentModal';
import ApplicationPreviewModal from '../../../../../components/modal/ApplicationPreviewModal';

const JobApplications = () => {
    const dispatch = useDispatch();
    const params = useParams();
    const jobId = params.id;

    const [storeApplicationStatus, setStoreApplicationStatus] = useState<
        Array<any>
    >([]);
    const [queryParams, setQueryParams] = useState<any>({
        status: 'unlisted',
        job_id: jobId,
    });
    const [statusQueryParams, setStatusQueryParams] = useState<any>({
        job_id: jobId,
        process_id: '1',
        action: 'load',
    });

    const getUser: object = useSelect(
        (select) => select(generalSettings).getCurrentUser(),
        []
    );
    const jobApplicationList: Array<any> = useSelect(
        (select) => select(jobstore).getJobApplicationList(queryParams),
        []
    );
    const jobApplicationTotal: any = useSelect(
        (select) => select(jobstore).getJobApplicationTotalCount(),
        []
    );
    const jobApplicationStatus: Array<any> = useSelect(
        (select) =>
            select(jobstore).getJobApplicationStatusCount(statusQueryParams),
        []
    );
    const jobApplicationLoading: boolean = useSelect(
        (select) => select(jobstore).getJobApplicationLoading(),
        []
    );

    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    const processedStatus = Object.values(jobApplicationStatus).slice(0, -2);
    const nexStatusData = jobApplicationStatus?.next_step;
    const prevStatusData = jobApplicationStatus?.previous_step;
    const processedApplication = {
        applications: jobApplicationList,
        prevStatus: prevStatusData,
        nextStatus: nexStatusData,
    };

    useEffect(() => {
        if (processedStatus?.length > 0) {
            setStoreApplicationStatus(processedStatus);
        }
    }, [jobApplicationStatus]);

    useEffect(() => {
        dispatch(jobstore).setJobApplications(queryParams);
        dispatch(jobstore).setJobApplicationStatus(statusQueryParams);
    }, [jobId, queryParams, statusQueryParams]);

    const applicationStatusHandler = (
        applicationId: string,
        moveProcessId: string,
        currentProcessId: string
    ) => {
        if (
            Number(currentProcessId) > 2 &&
            Number(moveProcessId) > Number(currentProcessId)
        ) {
            setStatusProcceedData({
                id: applicationId,
                movedProccessId: moveProcessId,
                currentProccessId: currentProcessId,
            });
            setStatusCommentModalOpen(true);
            const applicationDetails = getApplicationDetailsWithStatusById(
                processedApplication,
                applicationId
            );
            setJobSingleApplication(applicationDetails?.application);
        } else {
            dispatch(jobstore)
                .updateJobApplicationStatus({
                    id: applicationId,
                    process_id: moveProcessId,
                })
                .then((res: any) => {
                    if (res) {
                        setStatusQueryParams({
                            ...statusQueryParams,
                        });
                        for (
                            let i = 0;
                            i < storeApplicationStatus.length;
                            i++
                        ) {
                            if (storeApplicationStatus[i].id == moveProcessId) {
                                const count = Number(
                                    storeApplicationStatus[i].count
                                );
                                storeApplicationStatus[i].count = count + 1;
                            }
                            if (
                                storeApplicationStatus[i].id == currentProcessId
                            ) {
                                const count = Number(
                                    storeApplicationStatus[i].count
                                );
                                storeApplicationStatus[i].count = count - 1;
                            }
                        }
                    }
                })
                .catch((err: any) => {
                    console.error(err);
                });
        }
    };

    const applicationSearchHandler = lodash.debounce((text: string) => {
        setQueryParams({
            ...queryParams,
            search: text,
            page: 1,
        });
    }, 500);

    const [isPreviewModalOpen, setIsPreviewModalOpen] = useState(false);
    const [statusCommentModalOpen, setStatusCommentModalOpen] = useState(false);
    const [jobSingleApplication, setJobSingleApplication] = useState<any>({});
    const [applicationComments, setApplicationComments] = useState<Array<any>>(
        []
    );
    const [commentText, setCommentText] = useState<string>('');
    const [commentFormError, setCommentFormError] = useState<string>('');
    const [statusProcceedData, setStatusProcceedData] = useState<object>({});

    function closePreviewModal() {
        setIsPreviewModalOpen(false);
        setCommentText('');
    }
    function closeStatusCommentModal() {
        setStatusCommentModalOpen(false);
        setCommentText('');
    }

    const applicantPreviewHandler = (applicationId: string, isRead: string) => {
        setIsPreviewModalOpen(true);
        const data = {
            id: applicationId,
            is_read: '1',
        };

        if (isRead == '0') {
            dispatch(jobstore)
                .updateJobApplicationStatus(data)
                .then((res: any) => {
                    if (res) {
                        setQueryParams({ ...queryParams });
                    }
                })
                .catch((err: any) => {
                    console.error(err);
                });
        }
        const applicationDetails = getApplicationDetailsWithStatusById(
            processedApplication,
            applicationId
        );
        setJobSingleApplication(applicationDetails?.application);

        // Get Comments
        dispatch(jobstore)
            .setApplicationCommentById({
                app_id: applicationId,
            })
            .then((res: any) => {
                if (res) {
                    setApplicationComments(res);
                }
            })
            .catch((err: any) => {
                console.error(err);
            });
    };

    const submitCommentHandler = (e: any) => {
        e.preventDefault();
        const prepareComment = {
            user_id: getUser.id || null,
            app_id: jobSingleApplication.id || null,
            app_process_id: jobSingleApplication.process_id || null,
            comment: commentText,
        };
        if (commentText.length < 1) {
            setCommentFormError(__('Number Of Applicants', 'ai-job-listing'));
        } else {
            setCommentFormError('');
            dispatch(jobstore)
                .saveApplicationComment(prepareComment)
                .then((res: any) => {
                    if (res) {
                        setApplicationComments([res, ...applicationComments]);
                        setCommentText('');

                        if (statusProcceedData.id == jobSingleApplication.id) {
                            dispatch(jobstore)
                                .updateJobApplicationStatus({
                                    id: statusProcceedData.id,
                                    process_id:
                                        statusProcceedData.movedProccessId,
                                })
                                .then((statusRes: any) => {
                                    if (statusRes) {
                                        setStatusQueryParams({
                                            ...statusQueryParams,
                                        });
                                        for (
                                            let i = 0;
                                            i < storeApplicationStatus.length;
                                            i++
                                        ) {
                                            if (
                                                storeApplicationStatus[i].id ==
                                                statusProcceedData.movedProccessId
                                            ) {
                                                const count = Number(
                                                    storeApplicationStatus[i]
                                                        .count
                                                );
                                                storeApplicationStatus[
                                                    i
                                                ].count = count + 1;
                                            }
                                            if (
                                                storeApplicationStatus[i].id == statusProcceedData.currentProccessId
                                            ) {
                                                const count = Number(
                                                    storeApplicationStatus[i]
                                                        .count
                                                );
                                                storeApplicationStatus[
                                                    i
                                                ].count = count - 1;
                                            }
                                        }
                                    }
                                })
                                .catch((err: any) => {
                                    console.error(err);
                                });
                            setStatusProcceedData({});
                            setStatusCommentModalOpen(false);
                        }
                    }
                })
                .catch((err: any) => {
                    // eslint-disable-next-line no-console
                    console.error(err);
                });
        }
    };

    const [statusChanging, setStatusChanging] = useState<boolean>(false);

    const applicationPaginationHandler = (pages: number) => {
        if (pages != null) {
            setQueryParams({
                ...queryParams,
                page: pages,
            });
        }
    };

    const initialApplicationTableHeader = [
        {
            key: 'name',
            title: __('Name', 'ai-job-listing'),
            visibility: true,
        },
        {
            key: 'quick_preview',
            title: __('Quick Preview', 'ai-job-listing'),
            visibility: true,
        },
        {
            key: 'quick_actions',
            title: __('Quick Actions', 'ai-job-listing'),
            visibility: true,
        },
        {
            key: 'institute',
            title: __('Institute', 'ai-job-listing'),
            visibility: true,
        },
        {
            key: 'experience',
            title: __('Experience', 'ai-job-listing'),
            visibility: true,
            class: 'text-center',
            bodyClass: 'text-center',
        },
        {
            key: 'expected_salary',
            title: __('Expected Salary', 'ai-job-listing'),
            class: 'text-right',
            bodyClass: 'text-right',
            visibility: true,
        },
        {
            key: 'application_date',
            title: __('Application Date', 'ai-job-listing'),
            class: 'text-right',
            bodyClass: 'text-right',
            visibility: true,
        },
    ];
    const applicationTableHeader = initialApplicationTableHeader.filter(
        (header) => header.visibility
    );

    return (
        <>
            {/* Status Area */}
            <div className="flex flex-wrap gap-2.5 mb-5">
                {storeApplicationStatus?.map((status, index) => (
                    <div className="flex items-center gap-2.5" key={status.key}>
                        <button
                            style={{
                                backgroundColor:
                                    queryParams.status === status.key
                                        ? status.iconBg
                                        : '#ffffff',
                                borderColor:
                                    queryParams.status === status.key
                                        ? status.iconBg
                                        : '#CCCEDE',
                            }}
                            className={`${
                                queryParams.status === status.key
                                    ? 'pointer-events-none'
                                    : ''
                            } flex items-center gap-2.5 py-[8px] px-[25px] border-2 rounded-md`}
                            onClick={() => {
                                setQueryParams({
                                    ...queryParams,
                                    status: status.key,
                                    page: 1,
                                });
                                setStatusQueryParams({
                                    ...statusQueryParams,
                                    process_id: status.id ? status.id : '',
                                    action: 'click',
                                });
                                setStatusChanging(true);
                                setTimeout(() => {
                                    setStatusChanging(false);
                                }, 100);
                            }}
                        >
                            <span
                                style={{
                                    backgroundColor:
                                        queryParams.status === status.key
                                            ? '#ffffff'
                                            : status.iconBg,
                                    color:
                                        queryParams.status === status.key
                                            ? status.iconBg
                                            : '#ffffff',
                                }}
                                className={`h-[33px] w-[33px] flex items-center justify-center rounded-full text-white text-[13px]`}
                            >
                                <i className={status.icon}></i>
                            </span>
                            <span
                                className={`${
                                    queryParams.status === status.key
                                        ? 'text-white'
                                        : 'text-b1-notice-comment'
                                } text-[16px]`}
                            >
                                {status.label} {`(${status.count})`}
                            </span>
                        </button>
                        {storeApplicationStatus.length - 1 > index ? (
                            <span className="text-[15px] text-b2-table">
                                <i className="tio-chevron_right"></i>
                            </span>
                        ) : null}
                    </div>
                ))}
            </div>
            {/* Applications Filter Bar */}
            <FilterBar
                leftItem={
                    <>
                        <SearchInput
                            placeholder={__('Search Applicants...', 'ai-job-listing')}
                            onChange={(text) => {
                                applicationSearchHandler(text);
                            }}
                        />
                    </>
                }
            />
            {/* Applications Table Start*/}
            {jobApplicationLoading ? (
                <LoadingBox
                    block={processedApplication?.applications?.length}
                />
            ) : (
                <Table
                    header={applicationTableHeader}
                    body={processedApplication?.applications}
                >
                    <div customKey="name">
                        {(applicant: any) => (
                            <button
                                type="button"
                                className="flex items-center"
                                onClick={() =>
                                    applicantPreviewHandler(
                                        applicant.id,
                                        applicant.is_read
                                    )
                                }
                            >
                                {applicant.profile_image_url ? (
                                    <img
                                        className="w-[42px] h-[42px] m-0 rounded-full"
                                        src={applicant.profile_image_url}
                                        alt={getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'full_name'
                                        )}
                                    />
                                ) : (
                                    <span className=" rounded-full text-[#c5c5c5] text-[42px]">
                                        <i className="tio-account_circle"></i>
                                    </span>
                                )}
                                <span
                                    className={`ml-2.5 mb-0 text-[15px] font-medium ${
                                        applicant.is_read == '0'
                                            ? 'text-primary font-bold'
                                            : 'text-b1-notice-comment'
                                    }`}
                                >
                                    {getApplicantValueByKey(
                                        applicant?.meta_attributes,
                                        'full_name'
                                    )}
                                </span>
                            </button>
                        )}
                    </div>
                    <div customKey="quick_preview">
                        {(applicant: any) => (
                            <div className="inline-grid gap-2.5">
                                {applicant.resume && (
                                    <Button
                                        href={applicant.resume}
                                        text={__('View Resume', 'ai-job-listing')}
                                        btnStyle="action"
                                        icon="tio-file_outlined"
                                        outline
                                        externalLink
                                    />
                                )}
                                {getApplicantValueByKey(
                                    applicant?.meta_attributes,
                                    'protfolio_link'
                                ) && (
                                    <Button
                                        href={getApplicantValueByKey(
                                            applicant?.meta_attributes,
                                            'protfolio_link'
                                        )}
                                        text= {__('View Portfolio', 'ai-job-listing')} 
                                        btnStyle="action"
                                        icon="tio-link"
                                        outline
                                        externalLink
                                    />
                                )}
                            </div>
                        )}
                    </div>
                    <div customKey="quick_actions">
                        {(applicant: any) =>
                            applicant.next_step && applicant.previous_step ? (
                                <div>
                                    {applicant.process_id !=
                                        applicant?.previous_step?.id &&
                                        applicant.status == 'rejected' && (
                                            <Button
                                                text={__('Undo', 'ai-job-listing')}
                                                btnStyle="transparentDanger"
                                                icon="tio-clear"
                                                onClick={(e: any) =>
                                                    applicationStatusHandler(
                                                        applicant.id,
                                                        applicant.previous_process_id,
                                                        applicant.process_id
                                                    )
                                                }
                                            />
                                        )}
                                    {applicant.process_id !=
                                        applicant?.previous_step?.id &&
                                        applicant.status != 'rejected' && (
                                            <Button
                                                text={__('Undo', 'ai-job-listing')}
                                                btnStyle="transparentDanger"
                                                icon="tio-clear"
                                                onClick={(e: any) =>
                                                    applicationStatusHandler(
                                                        applicant.id,
                                                        applicant?.previous_step
                                                            ?.id,
                                                        applicant.process_id
                                                    )
                                                }
                                            />
                                        )}
                                    {applicant.process_id !=
                                        applicant?.next_step?.id && (
                                        <Button
                                            text={`${__('Move to', 'ai-job-listing')} ${applicant?.next_step?.name}`}
                                            btnStyle="actionSecondary"
                                            icon="tio-format_bullets"
                                            outline
                                            onClick={(e: any) =>
                                                applicationStatusHandler(
                                                    applicant.id,
                                                    applicant?.next_step?.id,
                                                    applicant.process_id
                                                )
                                            }
                                        />
                                    )}
                                    {applicant.process_id !=
                                        applicant?.next_step?.id && (
                                        <Button
                                            text= {__('Reject', 'ai-job-listing')}
                                            btnStyle="transparentDanger"
                                            icon="tio-clear"
                                            onClick={(e: any) =>
                                                applicationStatusHandler(
                                                    applicant.id,
                                                    applicant?.next_step
                                                        ?.reject_status_id,
                                                    applicant.process_id
                                                )
                                            }
                                        />
                                    )}
                                </div>
                            ) : (
                                <div>
                                    {applicant.process_id !=
                                        processedApplication?.prevStatus?.id &&
                                        applicant.status == 'rejected' && (
                                            <Button
                                                text={__('Undo', 'ai-job-listing')}
                                                btnStyle="transparentDanger"
                                                icon="tio-clear"
                                                onClick={(e: any) =>
                                                    applicationStatusHandler(
                                                        applicant.id,
                                                        applicant.previous_process_id,
                                                        applicant.process_id
                                                    )
                                                }
                                            />
                                        )}
                                    {applicant.process_id !=
                                        processedApplication?.prevStatus?.id &&
                                        applicant.status != 'rejected' && (
                                            <Button
                                                text={__('Undo', 'ai-job-listing')}
                                                btnStyle="transparentDanger"
                                                icon="tio-clear"
                                                onClick={(e: any) =>
                                                    applicationStatusHandler(
                                                        applicant.id,
                                                        processedApplication
                                                            ?.prevStatus?.id,
                                                        applicant.process_id
                                                    )
                                                }
                                            />
                                        )}
                                    {applicant.process_id !=
                                        processedApplication?.nextStatus
                                            ?.id && (
                                        <Button
                                            text={`${__('Move to', 'ai-job-listing')} ${processedApplication?.nextStatus?.name}`}
                                            btnStyle="actionSecondary"
                                            icon="tio-format_bullets"
                                            outline
                                            onClick={(e: any) =>
                                                applicationStatusHandler(
                                                    applicant.id,
                                                    processedApplication
                                                        ?.nextStatus?.id,
                                                    applicant.process_id
                                                )
                                            }
                                        />
                                    )}
                                    {applicant.process_id !=
                                        processedApplication?.nextStatus
                                            ?.id && (
                                        <Button 
                                            text={__('Reject', 'ai-job-listing')}
                                            btnStyle="transparentDanger"
                                            icon="tio-clear"
                                            onClick={(e: any) =>
                                                applicationStatusHandler(
                                                    applicant.id,
                                                    processedApplication
                                                        ?.nextStatus
                                                        ?.reject_status_id,
                                                    applicant.process_id
                                                )
                                            }
                                        />
                                    )}
                                </div>
                            )
                        }
                    </div>
                    <div customKey="institute">
                        {(applicant: any) => (
                            <>
                                <div className="mb-2.5 pb-2.5 border-b border-shadelight">
                                    {getApplicantValueByKey(
                                        applicant?.meta_attributes,
                                        'education_institute'
                                    )}
                                </div>
                                <div>
                                    {getApplicantValueByKey(
                                        applicant?.meta_attributes,
                                        'current_company_name'
                                    )}
                                </div>
                            </>
                        )}
                    </div>
                    <div customKey="experience">
                        {(applicant: any) =>
                            getApplicantValueByKey(
                                applicant?.meta_attributes,
                                'work_experience'
                            ) && (
                                <>
                                    {getApplicantValueByKey(
                                        applicant?.meta_attributes,
                                        'work_experience'
                                    )}
                                </>
                            )
                        }
                    </div>
                    <div customKey="expected_salary">
                        {(applicant: any) =>
                            getApplicantValueByKey(
                                applicant?.meta_attributes,
                                'expected_salary'
                            ) && (
                                <>
                                    {settings.currency_symbol}
                                    {getApplicantValueByKey(
                                        applicant?.meta_attributes,
                                        'expected_salary'
                                    )}
                                </>
                            )
                        }
                    </div>
                    <div customKey="application_date">
                        {(applicant: any) => formatDateOnly(applicant?.date)}
                    </div>
                </Table>
            )}

            {/* Applications Table End*/}
            {/* Pagination Area Start */}
            {statusChanging ? (
                ''
            ) : (
                <Pagination
                    perPage={settings?.posts_per_page}
                    total={jobApplicationTotal}
                    paginate={(pages: number) =>
                        applicationPaginationHandler(pages)
                    }
                />
            )}
            {/* Applicant Details Modal Start */}
            <ApplicationPreviewModal
                isPreviewModalOpen={isPreviewModalOpen}
                closePreviewModal={closePreviewModal}
                jobSingleApplication={jobSingleApplication}
                commentText={commentText}
                commentFormError={commentFormError}
                setCommentText={(text) => setCommentText(text)}
                submitCommentHandler={submitCommentHandler}
                applicationComments={applicationComments}
            />
            {/* Applicant Status Change Comments Modal Start */}
            <ApplicationCommentModal
                isModalOpen={statusCommentModalOpen}
                isModalClose={closeStatusCommentModal}
                onSubmit={submitCommentHandler}
                commentValue={commentText}
                inputError={commentFormError}
                onChange={(text) => setCommentText(text)}
            />
        </>
    );
};

export default JobApplications;
