/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useParams } from 'react-router-dom';
import {
    LineChart,
    Line,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
} from 'recharts';

/**
 * Internal dependencies.
 */
import jobstore from '../../../../../data/job';
import { getAllDatesBetweenCreateToDeadline } from '../../../../../utils/DateHelper';
import {
    getApplicationCountByDate,
    mergeTwoArray,
} from '../../../../../utils/Global';

const JobReports = ({ getJobDetails = {} }: any) => {
    const params = useParams();
    const jobId = params.id;
    const dispatch = useDispatch();

    const applicationDates: Array<any> = useSelect(
        (select) => select(jobstore).getJobApplicationDates(),
        []
    );

    useEffect(() => {
        dispatch(jobstore).setJobApplicationDateList({ job_id: jobId });
    }, []);

    const applicationCount = getApplicationCountByDate(applicationDates);
    const deadlineToCreateDate = getAllDatesBetweenCreateToDeadline(
        getJobDetails?.created_at,
        getJobDetails?.application_deadline
    );
    const generateDataForChart = mergeTwoArray(
        deadlineToCreateDate,
        applicationCount
    );

    const CustomTooltip = ({ active, payload, label }: any) => {
        if (active && payload && payload.length) {
            return (
                <div className="bg-primary rounded p-4 text-[15px] font-bold text-white">
                    <span className="block mb-1">{label}</span>
                    <span className="flex items-center gap-[3px]">
                        <span className="h-3 w-3 bg-white rounded-[2px]"></span>
                        <span className="mt-[1px]">{payload[0].value}</span>
                    </span>
                </div>
            );
        }
        return null;
    };

    return (
        <>
            <div className="flex items-center gap-5 mb-5">
                <div className="flex-1 flex items-center px-5 py-[14px] gap-[15px] border-2 border-border-mid rounded bg-white">
                    <div className="border-r border-border-mid pr-[15px]">
                        <span className="text-[30px] text-info">
                            <i className="tio-group_senior"></i>
                        </span>
                    </div>
                    <div className="flex-1 leading-[1.5]">
                        <div className="text-[20px] font-bold text-headline">
                            {getJobDetails?.total_applicants}
                        </div>
                        <span className="text-[16px] text-b1-notice-comment">
                        {__('Total Applications', 'ai-job-listing')}
                        </span>
                    </div>
                </div>
                <div className="flex-1 flex items-center px-5 py-[14px] gap-[15px] border-2 border-border-mid rounded bg-white">
                    <div className="border-r border-border-mid pr-[15px]">
                        <span className="text-[30px] text-warning">
                            <i className="tio-refresh"></i>
                        </span>
                    </div>
                    <div className="flex-1 leading-[1.5]">
                        <div className="text-[20px] font-bold text-headline">
                            {getJobDetails?.total_in_progress}
                        </div>
                        <span className="text-[16px] text-b1-notice-comment">
                              {__('In Progress', 'ai-job-listing')}
                        </span>
                    </div>
                </div>
                <div className="flex-1 flex items-center px-5 py-[14px] gap-[15px] border-2 border-border-mid rounded bg-white">
                    <div className="border-r border-border-mid pr-[15px]">
                        <span className="text-[30px] text-success">
                            <i className="tio-checkmark_circle"></i>
                        </span>
                    </div>
                    <div className="flex-1 leading-[1.5]">
                        <div className="text-[20px] font-bold text-headline">
                            {getJobDetails?.total_hired}
                        </div>
                        <span className="text-[16px] text-b1-notice-comment">
                            {__('Total Hired', 'ai-job-listing')}
                        </span>
                    </div>
                </div>
                <div className="flex-1 flex items-center px-5 py-[14px] gap-[15px] border-2 border-border-mid rounded bg-white">
                    <div className="border-r border-border-mid pr-[15px]">
                        <span className="text-[30px] text-red">
                            <i className="tio-clear_circle"></i>
                        </span>
                    </div>
                    <div className="flex-1 leading-[1.5]">
                        <div className="text-[20px] font-bold text-headline">
                            {getJobDetails?.total_rejected}
                        </div>
                        <span className="text-[16px] text-b1-notice-comment">
                        {__('Total Rejected', 'ai-job-listing')}
                        </span>
                    </div>
                </div>
            </div>
            <div className="bg-white p-8 rounded">
                <h3>{__('Number Of Applicants', 'ai-job-listing')}</h3>
                <div className="w-full h-[430px]">
                    <ResponsiveContainer width="100%" height="100%">
                        <LineChart
                            width={500}
                            height={300}
                            data={generateDataForChart}
                            margin={{ bottom: 25, top: 10, left: -40 }}
                        >
                            <CartesianGrid
                                stroke="#CCCEDE"
                                strokeDasharray="6 6"
                                vertical={false}
                            />
                            <XAxis
                                dataKey="date"
                                axisLine={{ stroke: '#323DA5' }}
                                tickLine={false}
                                domain={['dataMin', 'dataMax']}
                                tick={({ x, y, payload }) => {
                                    const isFirstTick = payload.index === 0;
                                    return (
                                        <text
                                            x={x} // Horizontal padding
                                            y={y + 25} // Vertical padding
                                            textAnchor="middle"
                                            fontSize={15}
                                            fontWeight={700}
                                            fill="#A8ABBD"
                                            dx={isFirstTick ? 20 : -10}
                                        >
                                            {payload.value}
                                        </text>
                                    );
                                }}
                                padding={{ right: 5, left: 20 }}
                            />
                            <YAxis
                                allowDecimals={false}
                                axisLine={false}
                                tickLine={false}
                                tick={({ payload, x, y }) => {
                                    const isFirstTick =
                                        payload.coordinate === 0;
                                    return (
                                        <text
                                            x={x - 10}
                                            y={y}
                                            textAnchor="start"
                                            fontSize={15}
                                            fontWeight={700}
                                            fill="#A8ABBD"
                                            dy={isFirstTick ? -10 : 4}
                                        >
                                            {payload.value}
                                        </text>
                                    );
                                }}
                            />
                            <Tooltip content={<CustomTooltip />} />
                            <Line
                                type="monotone"
                                dataKey="application"
                                stroke="#323DA5"
                                activeDot={{ r: 6 }}
                            />
                        </LineChart>
                    </ResponsiveContainer>
                </div>
            </div>
        </>
    );
};

export default JobReports;
