/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useSelect, useDispatch } from '@wordpress/data';
import { useNavigate, useParams } from 'react-router-dom';

/**
 * Internal dependencies.
 */

import generalSettings from '../../../../../data/general-settings';
import Button from '../../../../../components/button/Button';
import jobstore from '../../../../../data/job';
import { IJobPostForm } from '../../../../../data/job/state';
import { parseHtml } from '../../../../../utils/ParseHtml';
import { getValueByKey } from '../../../../../utils/Global';
import { formatDateOnly } from '../../../../../utils/DateHelper';
import { SwalSuccess } from '../../../../../utils/Swal';

const FormStepFive = () => {
    const params = useParams();
    const postId = params.id;

    const dispatch = useDispatch();
    const navigate = useNavigate();
    const getJobFormData: IJobPostForm = useSelect(
        (select) => select(jobstore).getJobPostForm(),
        []
    );
    const jobPostFormLoading: boolean = useSelect(
        (select) => select(jobstore).getJobPostFormLoading(),
        []
    );
    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );
    const onSubmit = () => {
        dispatch(jobstore)
            .saveJobPostForm({
                ...getJobFormData,
                form_step_complete: 5,
                status: 'active',
            })
            .then((result: any) => {
                if (result.status === 'active') {
                    if (postId) {
                        SwalSuccess({
                            title: __('Updated', 'ai-job-listing'),
                            text: __('Job Post Updated Successfully', 'ai-job-listing'),
                            position: 'center',
                        });
                    } else {
                        SwalSuccess({
                            title: __('Submitted', 'ai-job-listing'),
                            text:__('Job Post Submitted Successfully', 'ai-job-listing'),
                            position: 'center',
                        });
                    }
                    navigate('/job/thank-you', {
                        state: { ...getJobFormData, liveUrlBase: settings?.page_slug },
                    });
                }
            });
    };

    return (
        <>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary">
                Job Post Preview: {getJobFormData.title}
            </h4>
            {getJobFormData?.description && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Job Description', 'ai-job-listing')}</h3>
                    {parseHtml(getJobFormData?.description)}
                </div>
            )}
            {getJobFormData?.responsibilities && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Job Responsibilities', 'ai-job-listing')}</h3>
                    {parseHtml(getJobFormData?.responsibilities)}
                </div>
            )}
            {getJobFormData?.requirements && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Primary Requirements', 'ai-job-listing')}</h3>
                    {parseHtml(getJobFormData?.requirements)}
                </div>
            )}
            {getJobFormData?.experienced_year && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Experience', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getJobFormData?.experienced_year_label}
                    </div>
                </div>
            )}
            {getJobFormData?.additional_requirements && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Optional/Preferred Requirements', 'ai-job-listing')}</h3>
                    <div className="text-[16px] font-bold text-headline mb-2.5"> 
                        {__('***Candidates who meet the following requirements will be given preference.', 'ai-job-listing')} 
                    </div>
                    {parseHtml(getJobFormData?.additional_requirements)}
                </div>
            )}
            {getJobFormData?.benefits && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Benifits', 'ai-job-listing')}</h3>
                    {parseHtml(getJobFormData?.benefits)}
                </div>
            )}
            {getJobFormData?.additional_notes && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Additional Notes', 'ai-job-listing')}</h3>
                    {parseHtml(getJobFormData?.additional_notes)}
                </div>
            )}
            {getJobFormData?.job_locations && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Job Location', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getValueByKey(
                            getJobFormData.attribute_values,
                            'job_locations'
                        )}
                    </div>
                </div>
            )}
            {getJobFormData?.job_education_qualifications && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Educational Requirements', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getValueByKey(
                            getJobFormData.attribute_values,
                            'job_education_qualifications'
                        )}
                    </div>
                </div>
            )}
            {getJobFormData?.no_of_vacancies && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('No. Of Vacancies', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getJobFormData?.no_of_vacancies}
                    </div>
                </div>
            )}
            {getJobFormData?.job_types && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Job Type', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getValueByKey(
                            getJobFormData.attribute_values,
                            'job_types'
                        )}
                    </div>
                </div>
            )}
            {getJobFormData?.job_departments && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Job Department', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {getValueByKey(
                            getJobFormData.attribute_values,
                            'job_departments'
                        )}
                    </div>
                </div>
            )}
            {getJobFormData?.min_salary != 0 &&
                getJobFormData?.max_salary != 0 &&
                getJobFormData?.job_salary_type && (
                    <div className="mb-[30px]">
                        <h3 className="mb-2.5">{__('Salary Range', 'ai-job-listing')}</h3>
                        <div className="text-b1">
                            {`${settings.currency_symbol}${
                                getJobFormData?.min_salary
                            } - ${settings.currency_symbol}${
                                getJobFormData?.max_salary
                            } (${getValueByKey(
                                getJobFormData.attribute_values,
                                'job_salary_type'
                            )})`}
                        </div>
                    </div>
                )}
            {getJobFormData?.is_required_cv_photo === '1' && (
                <div className="mb-[30px]">
                    <h3 className="mb-2.5">{__('Photograph', 'ai-job-listing')}</h3>
                    <ul>
                        <li className="text-b1">
                            {__('Photograph must be enclosed with the resume', 'ai-job-listing')} 
                        </li>
                    </ul>
                </div>
            )}
            {getJobFormData?.application_deadline && (
                <div>
                    <h3 className="mb-2.5">{__('Application Deadline', 'ai-job-listing')}</h3>
                    <div className="text-b1">
                        {formatDateOnly(getJobFormData?.application_deadline)}
                    </div>
                </div>
            )}
            <div className="mt-[30px] pt-[30px] flex items-center justify-between gap-2.5 border-t border-solid border-border-mid">
                <Button
                    type="button"
                    btnStyle="light"
                    size="large"
                    outline
                    text= {__('Back', 'ai-job-listing')}
                    onClick={() => dispatch(jobstore).setJobFormStep(4)}
                />
                <Button
                    type="button"
                    btnStyle="secondary"
                    size="large" 
                    text= {__('Submit Now', 'ai-job-listing')}
                    icon={jobPostFormLoading ? 'animate-spin tio-refresh' : ''}
                    disabled={jobPostFormLoading}
                    onClick={onSubmit}
                />
            </div>
        </>
    );
};

export default FormStepFive;
