/**
 * External dependencies
 */
import { useState } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';

/**
 * Internal dependencies.
 */
import { __ } from '@wordpress/i18n';
import { Input, IInputResponse } from '../../../../../components/form/Input';
import Button from '../../../../../components/button/Button';
import jobstore from '../../../../../data/job';
import { IJobPostForm } from '../../../../../data/job/state';
import InputLabel from '../../../../../components/form/InputLabel';

const FormStepFour = () => {
    const dispatch = useDispatch();
    const [formValidationErrors, setFormValidationErrors] = useState<any>({});
    const [imageUrl, setImageUrl] = useState(null);
    const getJobFormData: IJobPostForm = useSelect(
        (select) => select(jobstore).getJobPostForm(),
        []
    );
    const jobPostFormLoading: boolean = useSelect(
        (select) => select(jobstore).getJobPostFormLoading(),
        []
    );

    // Custom Uploader
    const openMediaUploader = () => {
        const mediaUploader = wp.media({
            title: __('Choose Image', 'ai-job-listing'),
            button: {
                text: __('Choose Image', 'ai-job-listing'),
            },
            multiple: false,
            library: {
                type: 'image',
            },
        });

        mediaUploader.on('select', () => {
            const attachment = mediaUploader
                .state()
                .get('selection')
                .first()
                .toJSON();
            if (attachment.type !== 'image') {
                // eslint-disable-next-line no-alert
                alert(__('Choose Image', 'ai-job-listing'));
                return;
            }
            setImageUrl(attachment.url);
            dispatch(jobstore).setJobPostForm({
                ...getJobFormData,
                feature_image: attachment.url,
            });
        });

        mediaUploader.open();
    };

    const clearMediaHandler = () => {
        setImageUrl(null);
        dispatch(jobstore).setJobPostForm({
            ...getJobFormData,
            feature_image: null,
        });
    };

    const onChange = (input: IInputResponse) => {
        // Onchange Form Field Validation
        Object.keys(formValidationErrors).forEach((fieldName) => {
            if (fieldName === input.name) {
                delete formValidationErrors[fieldName];
            }
        });
        // Set Job Post Form data to the store
        dispatch(jobstore).setJobPostForm({
            ...getJobFormData,
            [input.name]:
                typeof input.value === 'object'
                    ? input.value?.value
                    : input.value,
            feature_image: imageUrl,
        });
    };
    const onSubmit = () => {
        // Validate form fields here
        const errors: any = {};
        if (getJobFormData?.seo_title === '') {
            errors.seo_title = __('Seo Title is required', 'ai-job-listing');
        }
        if (getJobFormData?.seo_description === '') {
            errors.seo_description = __('Seo description is required', 'ai-job-listing');
        }

        // Check error and store error in state for showing error on display
        if (Object.keys(errors).length > 0) {
            setFormValidationErrors(errors);
            return;
        }
        // if all error resolve send Job Post Form data to store for Insert
        dispatch(jobstore)
            .saveJobPostForm({
                ...getJobFormData,
                form_step_complete:
                    Number(getJobFormData?.form_step_complete) > 4
                        ? getJobFormData?.form_step_complete
                        : 4,
            })
            .then((response: any) => {
                dispatch(jobstore).setJobFormStep(5);
            });
    };

    return (
        <>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary">
                {__('SEO Information', 'ai-job-listing')}
            </h4>
            <div className="mb-5">
                <Input
                    type="text"
                    label={__('Meta Title *', 'ai-job-listing')}
                    name="seo_title"
                    value={getJobFormData.seo_title}
                    onChange={onChange}
                    errorMessage={formValidationErrors.seo_title}
                />
            </div>
            <div className="mb-5">
                <Input
                    type="textarea"
                    label={__('Meta Description *', 'ai-job-listing')}
                    name="seo_description"
                    value={getJobFormData.seo_description}
                    onChange={onChange}
                    errorMessage={formValidationErrors.seo_description}
                />
            </div>
            <div className="mb-5">
                <InputLabel className="mb-[20px]"> 
                    {__('Upload Featured Image', 'ai-job-listing')}
                </InputLabel>
                <div className="p-10 border-2 border-dashed border-border-mid rounded text-center">
                    {!getJobFormData?.feature_image ? (
                        <button
                            onClick={openMediaUploader}
                            type="button"
                            className="text-b2-regular"
                        >
                            <span className="text-headline font-medium block mb-[5px]">
                            {__('To upload select featured image click here to browser your computer', 'ai-job-listing')} 
                            </span>
                            <span className="text-b3-meta">
                            {__('The maximum file size is (5 Mb) and the supported file types are .png, .jpg, .jpeg .gif', 'ai-job-listing')}    
                            </span>
                        </button>
                    ) : (
                        <img
                            src={getJobFormData?.feature_image}
                            alt="Attachment"
                            className="max-w-full mx-auto mb-[30px] mt-0"
                        />
                    )}
                    <div className="flex items-center justify-center gap-5">
                        {getJobFormData?.feature_image && (
                            <Button
                                type="button"
                                btnStyle="secondary"
                                outline 
                                text= {__('Remove', 'ai-job-listing')}
                                onClick={clearMediaHandler}
                            />
                        )}
                    </div>
                </div>
            </div>
            <div className="mt-[30px] flex items-center justify-between gap-2.5">
                <Button
                    type="button"
                    btnStyle="light"
                    size="large"
                    outline
                    text= {__('Back', 'ai-job-listing')}
                    onClick={() => dispatch(jobstore).setJobFormStep(3)}
                />
                <Button
                    type="button"
                    btnStyle="secondary"
                    size="large"
                    text= {__('Next', 'ai-job-listing')}
                    icon={jobPostFormLoading ? 'animate-spin tio-refresh' : ''}
                    disabled={jobPostFormLoading}
                    onClick={onSubmit}
                />
            </div>
        </>
    );
};

export default FormStepFour;
