/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useParams, useNavigate } from 'react-router-dom';
/**
 * Internal dependencies.
 */
import { __ } from '@wordpress/i18n';
import { Input, IInputResponse } from '../../../../../components/form/Input';
import DatePicker from '../../../../../components/form/DatePicker';
import InputLabel from '../../../../../components/form/InputLabel';
import Button from '../../../../../components/button/Button';
import jobstore from '../../../../../data/job';
import { IJobPostForm } from '../../../../../data/job/state';
import settingStore from '../../../../../data/settings';

const FormStepOne = () => {
    const params = useParams();
    const postId = params.id;
    const navigate = useNavigate();

    const [formValidationErrors, setFormValidationErrors] = useState<any>({});
    const dispatch = useDispatch();
    const getJobFormData: IJobPostForm = useSelect(
        (select) => select(jobstore).getJobPostForm(),
        []
    );
    const jobPostFormLoading: boolean = useSelect(
        (select) => select(jobstore).getJobPostFormLoading(),
        []
    );

    // Job Settings Data
    const jobSettings: any = useSelect(
        (select) => select(settingStore).getJobSettings(),
        []
    );

    // Job Post Form Onchange Action
    const onChange = (input: IInputResponse) => {
        // Onchange Form Field Validation
        Object.keys(formValidationErrors).forEach((fieldName) => {
            if (fieldName === input.name) {
                delete formValidationErrors[fieldName];
            }
        });

        // Set Job Post Form data to the store
        dispatch(jobstore).setJobPostForm({
            ...getJobFormData,
            [input.name]:
                typeof input.value === 'object'
                    ? input.value?.value
                    : input.value,
        });
    };
    // Job Post Form OnSubmit Action
    const onSubmit = () => {
        // Validate form fields here
        const errors: any = {};
        if (getJobFormData.title === null || !getJobFormData.title.trim()) {
            errors.title = __('Title is required', 'ai-job-listing');
        }
        if (getJobFormData.job_departments === null) {
            errors.job_departments = __('Job Department is required', 'ai-job-listing');
        }
        if (getJobFormData.job_types === null) {
            errors.job_types = __('Job Type is required', 'ai-job-listing');
        }
        if (getJobFormData.job_shifts === null) {
            errors.job_shifts = __('Job Shift is required', 'ai-job-listing');
        }
        if (getJobFormData.no_of_vacancies === null) {
            errors.no_of_vacancies = __('No of Vacancies is required', 'ai-job-listing');
        }
        if (getJobFormData.application_deadline === null) {
            errors.application_deadline = __('Application Deadline is required', 'ai-job-listing');
        }
        if (getJobFormData.is_required_cv_photo === null) {
            errors.is_required_cv_photo = __('Please select one', 'ai-job-listing');
        }

        // Check error and store error in state for showing error on display
        if (Object.keys(errors).length > 0) {
            setFormValidationErrors(errors);
            return;
        }
        // if all error resolve send Job Post Form data to store for Insert
        dispatch(jobstore)
            .saveJobPostForm({
                ...getJobFormData,
                form_step_complete:
                    Number(getJobFormData?.form_step_complete) > 1
                        ? getJobFormData?.form_step_complete
                        : 1,
            })
            .then((response: any) => {
                if (!postId && response) {
                    navigate(`/job/edit/${response.id}`);
                }
                dispatch(jobstore).setJobFormStep(2);
            });
    };

    return (
        <>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary"> 
                {__('Primary Job Information', 'ai-job-listing')}
            </h4>
            <div className="grid grid-cols-2 gap-5">
                <div>
                    <Input
                        type="text"
                        label={__('Job Title *', 'ai-job-listing')}
                        name="title"
                        value={getJobFormData.title}
                        onChange={onChange}
                        errorMessage={formValidationErrors.title}
                    />
                </div>
                <div>
                    <Input
                        type="select"
                        label={__('Job Department *', 'ai-job-listing')}
                        name="job_departments"
                        value={getJobFormData.job_departments}
                        options={jobSettings.departments}
                        onChange={onChange}
                        errorMessage={formValidationErrors.job_departments}
                    />
                </div>
                <div>
                    <Input
                        type="select"
                        label={__('Job Type *', 'ai-job-listing')}
                        name="job_types"
                        value={getJobFormData.job_types}
                        options={jobSettings.jobType}
                        onChange={onChange}
                        errorMessage={formValidationErrors.job_types}
                    />
                </div>
                <div>
                    <Input
                        type="select"
                        label={__('Job Shift *', 'ai-job-listing')}
                        name="job_shifts"
                        value={getJobFormData.job_shifts}
                        options={jobSettings.shifts}
                        onChange={onChange}
                        errorMessage={formValidationErrors.job_shifts}
                    />
                </div>
                <div>
                    <InputLabel>{__('Application Deadline *', 'ai-job-listing')}</InputLabel>
                    <DatePicker
                        defaultValue={getJobFormData.application_deadline}
                        onChange={onChange}
                        minDate={new Date()}
                        id="application_deadline"
                    />
                    {formValidationErrors.application_deadline && (
                        <div className="text-error text-b5-regular mt-1.5">
                            {formValidationErrors.application_deadline}
                        </div>
                    )}
                </div>
                <div>
                    <Input
                        type="number"
                        label={__('No. Of Vacancies *', 'ai-job-listing')}
                        name="no_of_vacancies"
                        value={getJobFormData.no_of_vacancies}
                        onChange={onChange}
                        errorMessage={formValidationErrors.no_of_vacancies}
                    />
                </div>
                <div>
                    <InputLabel>Require Applicant to Upload Photo *</InputLabel>
                    <div className="flex items-center gap-[10px]">
                        <div>
                            <label
                                htmlFor="resume_inclose_no"
                                className="block py-[8px] px-[20px] border border-solid border-border-dark rounded cursor-pointer"
                            >
                                <Input
                                    type="radio"
                                    name="is_required_cv_photo"
                                    id="resume_inclose_no"
                                    value="0"
                                    onChange={onChange}
                                    checked={
                                        getJobFormData.is_required_cv_photo === '0'
                                            ? true
                                            : undefined
                                    }
                                    right={
                                        <InputLabel
                                            htmlFor="resume_inclose_no"
                                            className="mb-0 ml-[10px] font-normal !text-b1-notice-comment cursor-pointer"
                                        >
                                            No
                                        </InputLabel>
                                    }
                                />
                            </label>
                        </div>
                        <div>
                            <label
                                htmlFor="resume_inclose_yes"
                                className="block py-[8px] px-[20px] border border-solid border-border-dark rounded cursor-pointer"
                            >
                                <Input
                                    type="radio"
                                    name="is_required_cv_photo"
                                    id="resume_inclose_yes"
                                    value="1"
                                    onChange={onChange}
                                    checked={
                                        getJobFormData.is_required_cv_photo === '1'
                                            ? true
                                            : undefined
                                    }
                                    right={
                                        <InputLabel
                                            htmlFor="resume_inclose_yes"
                                            className="mb-0 ml-[10px] font-normal !text-b1-notice-comment cursor-pointer"
                                        >
                                            Yes
                                        </InputLabel>
                                    }
                                />
                            </label>
                        </div>
                    </div>
                    {formValidationErrors.is_required_cv_photo && (
                        <div className="text-error text-b5-regular mt-1.5">
                            {formValidationErrors.is_required_cv_photo}
                        </div>
                    )}
                </div>
            </div>
            <div className="mt-[30px] text-right">
                <Button
                    type="button"
                    btnStyle="secondary"
                    size="large"
                    text="Next"
                    icon={jobPostFormLoading ? 'animate-spin tio-refresh' : ''}
                    disabled={jobPostFormLoading}
                    onClick={onSubmit}
                />
            </div>
        </>
    );
};

export default FormStepOne;
