/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';

/**
 * Internal dependencies.
 */
import { __ } from '@wordpress/i18n';
import { Input, IInputResponse } from '../../../../../components/form/Input';
import InputLabel from '../../../../../components/form/InputLabel';
import Button from '../../../../../components/button/Button';
import jobstore from '../../../../../data/job';
import { IJobPostForm } from '../../../../../data/job/state';
import settingStore from '../../../../../data/settings';

const FormStepThree = () => {
    const dispatch = useDispatch();
    const [formValidationErrors, setFormValidationErrors] = useState<any>({});
    const [addireqOpen, setAddireqOpen] = useState(false);
    const [addinoteOpen, setAddinoteOpen] = useState(false);
    const getJobFormData: IJobPostForm = useSelect(
        (select) => select(jobstore).getJobPostForm(),
        []
    );
    const jobPostFormLoading: boolean = useSelect(
        (select) => select(jobstore).getJobPostFormLoading(),
        []
    );

    // Job Settings Data
    const jobSettings: any = useSelect(
        (select) => select(settingStore).getJobSettings(),
        []
    );

    const onChange = (input: IInputResponse) => {
        // Onchange Form Field Validation
        Object.keys(formValidationErrors).forEach((fieldName) => {
            if (fieldName === input.name) {
                delete formValidationErrors[fieldName];
            }
        });

        // Set Job Post Form data to the store
        dispatch(jobstore).setJobPostForm({
            ...getJobFormData,
            [input.name]:
                typeof input.value === 'object'
                    ? input.value?.value
                    : input.value,
        });
    };
    const onSubmit = () => {
        // Validate form fields here
        const errors: any = {};
        if (getJobFormData?.job_education_qualifications === null) {
            errors.job_education_qualifications = 'Education is required';
        }
        if (getJobFormData?.requirements === '') {
            errors.requirements = 'Primary requirement is required';
        }
        if (
            getJobFormData?.experienced_year === '' ||
            getJobFormData?.experienced_year === null
        ) {
            errors.experienced_year = 'Experience is required';
        }
        if (addireqOpen) {
            if (getJobFormData?.additional_requirements === '') {
                errors.additional_requirements =
                    'Additional requirement is required';
            }
        }
        if (addinoteOpen) {
            if (getJobFormData?.additional_notes === '') {
                errors.additional_notes = 'Additional Note is required';
            }
        }

        // Check error and store error in state for showing error on display
        if (Object.keys(errors).length > 0) {
            setFormValidationErrors(errors);
            return;
        }
        // if all error resolve send Job Post Form data to store for Insert
        dispatch(jobstore)
            .saveJobPostForm({
                ...getJobFormData,
                additional_requirements: addireqOpen
                    ? getJobFormData.additional_requirements
                    : '',
                additional_notes: addinoteOpen
                    ? getJobFormData.additional_notes
                    : '',
                form_step_complete:
                    Number(getJobFormData?.form_step_complete) > 3
                        ? getJobFormData?.form_step_complete
                        : 3,
            })
            .then((response: any) => {
                dispatch(jobstore).setJobFormStep(4);
            });
    };

    useEffect(() => {
        // Checkbox field check that data present or not
        if (getJobFormData?.additional_requirements !== '') {
            setAddireqOpen(true);
        }
        if (getJobFormData?.additional_notes !== '') {
            setAddinoteOpen(true);
        }
    }, []);

    return (
        <>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary">
                {__('Educational Qualifications', 'ai-job-listing')}     
            </h4>
            <div className="max-w-[415px] mb-5">
                <Input
                    type="select"
                    label={__('Preferred *', 'ai-job-listing')}
                    name="job_education_qualifications"
                    value={getJobFormData.job_education_qualifications}
                    options={jobSettings.qualification}
                    onChange={onChange}
                    errorMessage={
                        formValidationErrors.job_education_qualifications
                    }
                />
            </div>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary"> 
                {__('Candidates Requirements', 'ai-job-listing')}  
            </h4>
            <div className="mb-5">
                <Input
                    type="text-editor"
                    placeholder="Primary Requirements"
                    label={__('Primary Requirements *', 'ai-job-listing')}
                    id="requirements"
                    value={getJobFormData.requirements}
                    onChange={onChange}
                    errorMessage={formValidationErrors.requirements}
                />
            </div>
            <div className="max-w-[415px] mb-5">
                <Input
                    type="select"
                    label={__('Experience *', 'ai-job-listing')}
                    name="experienced_year"
                    value={getJobFormData.experienced_year}
                    options={jobSettings.experinceYears}
                    onChange={onChange}
                    errorMessage={formValidationErrors.experienced_year}
                />
            </div>
            <div className="mb-5">
                <Input
                    type="checkbox"
                    id="additional-req"
                    onChange={() => {
                        setAddireqOpen(!addireqOpen);
                    }}
                    checked={addireqOpen}
                    right={
                        <InputLabel htmlFor="additional-req" className="mb-0">
                             {__('Optional/Preferred Requirements:', 'ai-job-listing')}  
                        </InputLabel>
                    }
                    style={{ marginRight: '8px' }}
                />
            </div>
            {addireqOpen && (
                <div className="mb-5">
                    <Input
                        type="text-editor"
                        placeholder="Additional Requirements"
                        id="additional_requirements"
                        value={getJobFormData.additional_requirements}
                        onChange={onChange}
                        errorMessage={
                            formValidationErrors.additional_requirements
                        }
                    />
                </div>
            )}
            <div className="mb-5">
                <Input
                    type="checkbox"
                    id="additional-note"
                    onChange={() => setAddinoteOpen(!addinoteOpen)}
                    checked={addinoteOpen}
                    right={
                        <InputLabel htmlFor="additional-note" className="mb-0"> 
                            {__('Additional Note:', 'ai-job-listing')}  
                        </InputLabel>
                    }
                    style={{ marginRight: '8px' }}
                />
            </div>
            {addinoteOpen && (
                <div className="mb-5">
                    <Input
                        type="text-editor"
                        placeholder="Additional Note"
                        id="additional_notes"
                        value={getJobFormData.additional_notes}
                        onChange={onChange}
                        errorMessage={formValidationErrors.additional_notes}
                    />
                </div>
            )}
            <div className="mt-[30px] flex items-center justify-between gap-2.5">
                <Button
                    type="button"
                    btnStyle="light"
                    size="large"
                    outline
                    text="Back"
                    onClick={() => dispatch(jobstore).setJobFormStep(2)}
                />
                <Button
                    type="button"
                    btnStyle="secondary"
                    size="large"
                    text="Next"
                    icon={jobPostFormLoading ? 'animate-spin tio-refresh' : ''}
                    disabled={jobPostFormLoading}
                    onClick={onSubmit}
                />
            </div>
        </>
    );
};

export default FormStepThree;
