/**
 * External dependencies
 */
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';

/**
 * Internal dependencies.
 */
import { Input, IInputResponse } from '../../../../../components/form/Input';
import InputLabel from '../../../../../components/form/InputLabel';
import Button from '../../../../../components/button/Button';
import jobstore from '../../../../../data/job';
import { IJobPostForm } from '../../../../../data/job/state';
import settingStore from '../../../../../data/settings';
import Modal from '../../../../../components/modal/Modal';
import { parseHtml } from '../../../../../utils/ParseHtml';
import LoaderLine from '../../../../../components/loader/LoaderLine';
import { fallbackCopyTextHandler } from '../../../../../utils/Global';
import generalSettings from '../../../../../data/general-settings';

const FormStepTwo = () => {
    const dispatch = useDispatch();
    const [formValidationErrors, setFormValidationErrors] = useState<any>({});
    const [benifitOpen, setBenifitOpen] = useState(false);
    const [isGeneratedTextCopied, setIsGeneratedTextCopied] = useState(false);
    const getJobFormData: IJobPostForm = useSelect(
        (select) => select(jobstore).getJobPostForm(),
        []
    );
    const jobPostFormLoading: boolean = useSelect(
        (select) => select(jobstore).getJobPostFormLoading(),
        []
    );
    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    // Job Settings Data
    const jobSettings: any = useSelect(
        (select) => select(settingStore).getJobSettings(),
        []
    );
    const jobAiDescription: any = useSelect(
        (select) => select(jobstore).getJobAiDescription(),
        []
    );
    const jobAiDescriptionLoading = useSelect(
        (select) => select(jobstore).getJobAiDescriptionLoading(),
        []
    );

    const [isOpen, setIsOpen] = useState(false);
    const [reGenerateText, setReGenerateText] = useState<any>('');
    function closeModal() {
        setIsOpen(false);
        setIsGeneratedTextCopied(false);
    }

    // Generate and Modal Open
    const jobDescriptionGenerateHandler = () => {
        if (getJobFormData.id && settings?.open_ai_api_key) {
            setIsOpen(true);
            dispatch(jobstore).generateAiDescription({ id: getJobFormData.id });
        }
    };

    // Re-generate Job description
    const reGenerateJobDescriptionHandler = (e: any) => {
        e.preventDefault();
        dispatch(jobstore).generateAiDescription({
            new_query_txt: reGenerateText,
        });
        setIsGeneratedTextCopied(false);
    };

    // Copy Generated Descripiton
    const jobDescriptionCopyHandler = (e: any) => {
        const regex = /(<([^>]+)>)/gi;
        const generatedTextWithoutHtmlTag = jobAiDescription.replace(regex, '');

        fallbackCopyTextHandler(generatedTextWithoutHtmlTag).then(
            (response: any) => {
                if (response?.message) {
                    setIsGeneratedTextCopied(true);
                }
            }
        );
    };

    const onChange = (input: IInputResponse) => {
        // Onchange Form Field Validation
        Object.keys(formValidationErrors).forEach((fieldName) => {
            if (fieldName === input.name) {
                delete formValidationErrors[fieldName];
            }
        });
        // Set Job Post Form data to the store
        dispatch(jobstore).setJobPostForm({
            ...getJobFormData,
            [input.name]:
                typeof input.value === 'object'
                    ? input.value?.value
                    : input.value,
        });
    };
    const onSubmit = () => {
        // Validate form fields here
        const errors: any = {};
        if (getJobFormData?.description === '') {
            errors.description = 'Description is required';
        }
        if (getJobFormData?.job_locations === null) {
            errors.job_locations = 'Location is required';
        }
        if (
            getJobFormData?.responsibilities === '' ||
            getJobFormData?.responsibilities === null
        ) {
            errors.responsibilities = 'Job Responsibility is required';
        }
        if (
            Number(getJobFormData?.max_salary) <
            Number(getJobFormData?.min_salary)
        ) {
            errors.max_salary = 'Max salary should be greater than min salary';
        }

        if (benifitOpen) {
            if (getJobFormData?.benefits === '') {
                errors.benefits = 'Benifit requirement is required';
            }
        }

        // Check error and store error in state for showing error on display
        if (Object.keys(errors).length > 0) {
            setFormValidationErrors(errors);
            return;
        }
        // if all error resolve send Job Post Form data to store for Insert
        dispatch(jobstore)
            .saveJobPostForm({
                ...getJobFormData,
                form_step_complete:
                    Number(getJobFormData?.form_step_complete) > 2
                        ? getJobFormData?.form_step_complete
                        : 2,
            })
            .then((response: any) => {
                dispatch(jobstore).setJobFormStep(3);
            });
    };

    useEffect(() => {
        // Checkbox field check that data present or not
        if (getJobFormData?.benefits !== '') {
            setBenifitOpen(true);
        }
        dispatch(generalSettings).getGeneralSettings();
    }, []);

    return (
        <>
            <h4 className="pb-5 mb-5 border-b border-border-mid text-primary">
                More Job Information
            </h4>
            <div className="text-right">
                <Button
                    text={
                        jobAiDescriptionLoading ? 'Proccessing' : 'AI Generator'
                    }
                    href={
                        !settings?.open_ai_api_key
                            ? '/settings/general/ai-settings'
                            : undefined
                    }
                    icon={
                        jobAiDescriptionLoading
                            ? 'animate-spin tio-refresh'
                            : 'tio-magic_wand'
                    }
                    disabled={jobAiDescriptionLoading ? true : false}
                    btnStyle="secondary"
                    onClick={jobDescriptionGenerateHandler}
                />
            </div>
            <div className="mb-5">
                <Input
                    type="text-editor"
                    placeholder="{__('Write Job Description', 'ai-job-listing')}"
                    label={__('Job Description *', 'ai-job-listing')}
                    id="description"
                    value={getJobFormData.description}
                    onChange={onChange}
                    errorMessage={formValidationErrors.description}
                />
            </div>
            <div className="max-w-[415px] mb-5">
                <Input
                    type="select"
                    label={__('Location *', 'ai-job-listing')}
                    name="job_locations"
                    value={getJobFormData.job_locations}
                    options={jobSettings.locations}
                    onChange={onChange}
                    errorMessage={formValidationErrors.job_locations}
                />
            </div>
            <div className="mb-5">
                <InputLabel>Salary</InputLabel>
                <div className="flex items-center gap-5 max-w-[500px]">
                    <div className="w-[130px]">
                        <Input
                            type="number"
                            placeholder="Min. Salary"
                            name="min_salary"
                            value={
                                getJobFormData.min_salary > 0
                                    ? getJobFormData.min_salary
                                    : ''
                            }
                            onChange={onChange}
                            // errorMessage={formValidationErrors.min_salary}
                        />
                    </div>
                    <div className="w-[130px]">
                        <Input
                            type="number"
                            placeholder="Max. Salary"
                            name="max_salary"
                            value={
                                getJobFormData.max_salary > 0
                                    ? getJobFormData.max_salary
                                    : ''
                            }
                            onChange={onChange}
                            // errorMessage={formValidationErrors.max_salary}
                        />
                    </div>
                    <div className="flex-1">
                        <Input
                            type="select"
                            name="job_salary_type"
                            value={getJobFormData.job_salary_type}
                            options={jobSettings.salaryType}
                            onChange={onChange}
                        />
                    </div>
                </div>
                <div className="text-error text-b5-regular mt-1.5">
                    {formValidationErrors.max_salary}
                </div>
            </div>
            <div className="mb-5">
                <Input
                    type="text-editor"
                    placeholder="Write Job Description"
                    label={__('Job Responsibility *', 'ai-job-listing')}
                    id="responsibilities"
                    value={getJobFormData.responsibilities}
                    onChange={onChange}
                    errorMessage={formValidationErrors.responsibilities}
                />
            </div>
            <div className="mb-5">
                <Input
                    type="checkbox"
                    id="job-benifit"
                    onChange={() => setBenifitOpen(!benifitOpen)}
                    checked={benifitOpen}
                    right={
                        <InputLabel htmlFor="job-benifit" className="mb-0">
                            Job Benefits
                        </InputLabel>
                    }
                    style={{ marginRight: '8px' }}
                />
            </div>
            {benifitOpen && (
                <div className="mb-5">
                    <Input
                        type="text-editor"
                        placeholder="Write Job Benifits"
                        id="benefits"
                        value={getJobFormData.benefits}
                        onChange={onChange}
                        errorMessage={formValidationErrors.benefits}
                    />
                </div>
            )}
            <div className="mt-[30px] flex items-center justify-between gap-2.5">
                <Button
                    type="button"
                    btnStyle="light"
                    size="large"
                    outline
                    text="Back"
                    onClick={() => dispatch(jobstore).setJobFormStep(1)}
                />
                <Button
                    type="button"
                    btnStyle="secondary"
                    size="large"
                    text="Next"
                    icon={jobPostFormLoading ? 'animate-spin tio-refresh' : ''}
                    disabled={jobPostFormLoading}
                    onClick={onSubmit}
                />
            </div>
            {/* Description Generate Modal */}
            <Modal
                isOpen={isOpen}
                closeModal={closeModal}
                areaClass="mt-10 max-w-[620px]"
            >
                <div className="flex justify-between items-end mb-5">
                    <h4 className="mb-0">Job Description</h4>
                    <button
                        onClick={closeModal}
                        className="text-[20px] text-border-dark leading-[1] absolute top-5 right-5 focus:outline-none focus:shadow-none"
                    >
                        <i className="tio-clear"></i>
                    </button>
                </div>
                <div>
                    <div className="mb-5 h-[240px] overflow-y-auto">
                        {jobAiDescriptionLoading ? (
                            <LoaderLine line={10} />
                        ) : (
                            parseHtml(jobAiDescription)
                        )}
                    </div>
                    {!jobAiDescriptionLoading && (
                        <button
                            type="button"
                            onClick={(e) => jobDescriptionCopyHandler(e)}
                            className="text-[16px] text-green leading-[1]"
                        >
                            {isGeneratedTextCopied ? (
                                'Copied!'
                            ) : (
                                <i className="tio-copy text-[20px] text-b1-notice-comment"></i>
                            )}
                        </button>
                    )}
                </div>
                <div className="mt-5">
                    <form onSubmit={(e) => reGenerateJobDescriptionHandler(e)}>
                        <div
                            className={`relative ${
                                jobAiDescriptionLoading
                                    ? 'pointer-events-none'
                                    : ''
                            }`}
                        >
                            <Input
                                name="attribute_search"
                                type="text"
                                value={reGenerateText}
                                placeholder="Message"
                                onChange={(text) =>
                                    setReGenerateText(text.value)
                                }
                                className="!pr-10"
                            />
                            <button
                                type="submit"
                                className="px-[6px] py-[5px] rounded-xsm absolute top-2 right-3 z-10 text-[14px] leading-[1] text-white bg-secondary"
                            >
                                {jobAiDescriptionLoading ? (
                                    <span className="animate-spin block">
                                        <i className="tio-sync"></i>
                                    </span>
                                ) : (
                                    <i className="tio-arrow_upward"></i>
                                )}
                            </button>
                        </div>
                    </form>
                </div>
            </Modal>
        </>
    );
};

export default FormStepTwo;
