/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */
import generalSettings from '../../../../data/general-settings';
import Button from '../../../../components/button/Button';
import { SwalSuccess } from '../../../../utils/Swal';
import SwitchCheckbox from '../../../../components/form/SwitchCheckbox';
import PageTitle from '../../../../components/layout/PageTitle';
import { inputStyle } from '../../../../components/form/StyleClass';
import InputLabel from '../../../../components/form/InputLabel';
import LoaderLine from '../../../../components/loader/LoaderLine';

const AiSettings = () => {
    const dispatch = useDispatch();
    const [allSettings, setAllSettings] = useState<any>([]);
    const settingsLoading: boolean = useSelect(
        (select) => select(generalSettings).getGeneralSettingsLoading(),
        []
    );
    const settingSaveLoading: any = useSelect(
        (select) => select(generalSettings).getGeneralSettingSaveLoading(),
        []
    );
    const [settingsField, setSettingsField] = useState<any>([]);
    const [settingValues, setSettingValues] = useState<any>([]);

    useEffect(() => {
        dispatch(generalSettings)
            .getGeneralSettingsField()
            .then((response) => {
                setAllSettings(response);
            });
    }, []);

    useEffect(() => {
        const settingsFieldFilter = allSettings.filter(
            (data: any) => data.type === 'ai_setting'
        );
        setSettingsField(settingsFieldFilter);
        const settingsInitiateValue = allSettings.map((settings: any) => {
            return {
                name: settings.name,
                value: settings.value,
            };
        });
        setSettingValues(settingsInitiateValue);
    }, [allSettings]);

    const generateFormField = (settings: any) => {
        if (
            settings.form_type === 'text' ||
            settings.form_type === 'number' ||
            settings.form_type === 'password'
        ) {
            return (
                <input
                    type={settings.form_type}
                    name={settings.name}
                    defaultValue={settings.value}
                    placeholder={settings.placeholder}
                    className={`!text-b2-table ${inputStyle}`}
                    onChange={(e) =>
                        settingsOnChange({
                            name: settings.name,
                            value: e.target.value,
                        })
                    }
                />
            );
        } else if (settings.form_type === 'color') {
            return (
                <div className="h-[42px] flex items-center justify-between border border-solid border-border-mid rounded px-[15px]">
                    <div className="flex items-center gap-2.5">
                        <input
                            type="color"
                            name={settings.name}
                            defaultValue={settings.value}
                            value={settings.value}
                            className="h-[25px] w-[25px] !rounded-[2px] !border-0 !ring-0 focus:border-secondary"
                            onChange={(e) => {
                                settings.value = e.target.value;
                                settingsOnChange({
                                    name: settings.name,
                                    value: e.target.value,
                                });
                            }}
                        />
                        <span className="text-[14px] text-b2-table leading-[1]">
                            {settings.value}
                        </span>
                    </div>
                    {((settings.name == 'primary_color' &&
                        settings.value != '#323da5') ||
                        (settings.name == 'secondary_color' &&
                            settings.value != '#ff7f5c')) && (
                        <Button
                            type="button"
                            text="Reset"
                            icon="tio-restore text-[16px]"
                            buttonCustomClass="text-placeholder text-[12px] font-medium"
                            onClick={() => {
                                settings.value =
                                    settings.name == 'primary_color'
                                        ? '#ff7f5c'
                                        : '#323da5';
                                settingsOnChange({
                                    name: settings.name,
                                    value:
                                        settings.name == 'primary_color'
                                            ? '#ff7f5c'
                                            : '#323da5',
                                });
                            }}
                        />
                    )}
                </div>
            );
        } else if (settings.form_type === 'switcher') {
            return (
                <SwitchCheckbox
                    parentClass="h-[42px] flex items-center border border-solid border-border-mid rounded px-[15px]"
                    id={settings.name}
                    initialValue={settings.value == 'true' ? true : false}
                    onChange={(val: any) => {
                        settingsOnChange({
                            name: settings.name,
                            value: JSON.stringify(val.value),
                        });
                    }}
                    label
                />
            );
        }
    };

    const settingsOnChange = (inputSettings: any) => {
        const modifySettingsValues = settingValues.map((settings: any) => {
            return {
                name: settings.name,
                value:
                    inputSettings.name === settings.name
                        ? inputSettings.value
                        : settings.value,
            };
        });
        setSettingValues(modifySettingsValues);
    };

    const settingsOnSubmit = () => {
        dispatch(generalSettings)
            .updateGeneralSettings({
                settings: settingValues,
            })
            .then((response: any) => {
                if (response) {
                    SwalSuccess({
                        title: __(
                            'Settings Save Successfully',
                            'ai-job-listing'
                        ),
                    });
                }
            });
    };
    
    return (
        <div className="max-w-[690px]">
            <PageTitle text="AI Settings" backButton />
            <div className="bg-white p-[30px] rounded">
                {settingsLoading ? (
                    <LoaderLine line={15} />
                ) : (
                    <>
                        <div className="grid grid-cols-2 gap-5">
                            {settingsField?.map((settings: any) => (
                                <div
                                    className={
                                        settings.column_width === 'full'
                                            ? 'col-span-2'
                                            : 'col-span-1'
                                    }
                                    key={settings.id}
                                >
                                    <InputLabel>{settings.label}</InputLabel>
                                    {generateFormField(settings)}
                                </div>
                            ))}
                        </div>
                        <div className="flex item-center gap-5 mt-[30px]">
                            <Button
                                type="button"
                                btnStyle="secondary"
                                text="Save Changes"
                                icon={
                                    settingSaveLoading &&
                                    'animate-spin tio-refresh'
                                }
                                onClick={settingsOnSubmit}
                            />
                        </div>
                    </>
                )}
            </div>
        </div>
    );
};

export default AiSettings;