/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { __ } from '@wordpress/i18n';
import { useParams } from 'react-router-dom';
const lodash = require('lodash');

/**
 * Internal dependencies.
 */
import settingStore from '../../../../data/settings';
import generalSettings from '../../../../data/general-settings';
import PageTitle from '../../../../components/layout/PageTitle';
import {
    ISettingsAttributes,
    IsettingsMetaForm,
    settingsMetaDefaultData,
} from '../../../../data/settings/state';
import { InputType } from '../../../../interfaces';
import FilterBar from '../../../../components/table/FilterBar';
import Button from '../../../../components/button/Button';
import Modal from '../../../../components/modal/Modal';
import LoadingBox from '../../../../components/loader/Loading';
import { SwalWarning } from '../../../../utils/Swal';
import { Input } from '../../../../components/form/Input';
import { selectStylesSM } from '../../../../components/form/StyleClass';
import Pagination from '../../../../components/pagination/Pagination';
import Table from '../../../../components/table/Table';
import Dropdown from '../../../../components/dropdown/Dropdown';
import { Menu } from '@headlessui/react';
import SearchInput from '../../../../components/form/SearchInput';
import { sortAlphabetically } from '../../../../utils/Global';

const JobSettingsAttributes = () => {
    const [isOpen, setIsOpen] = useState(false);
    const dispatch = useDispatch();
    const params = useParams();
    const attributeSlug = params.slug;
    const [trash, setTrash] = useState(false);
    const [attributeLabel, setAttributeLabel] = useState<string>('');
    const [settingsAttributeMeta, setSettingsAttributeMeta] = useState<
        Array<any>
    >([]);
    const [queryParams, setQueryParams] = useState<any>({
        attribute_slug: attributeSlug,
        status: 'active',
    });

    const settings: object = useSelect(
        (select) => select(generalSettings).getGeneralSettings(),
        []
    );

    function closeModal() {
        setIsOpen(false);
    }
    function openModal() {
        setIsOpen(true);
    }

    const settingsMetaFormData: IsettingsMetaForm = useSelect(
        (select) => select(settingStore).getMetaFormData(),
        [attributeSlug]
    );
    const loadingAttributeMeta: ISettingsAttributes = useSelect(
        (select) => select(settingStore).getLoadingSettingsMeta(),
        []
    );
    const settingAttributeMetaTotal: ISettingsAttributes = useSelect(
        (select) => select(settingStore).getSettingsAttributeMetaTotal(),
        []
    );

    useEffect(() => {
        dispatch(settingStore)
            .getSettingAttributeMetas(queryParams)
            .then((response: any) => {
                if (response) {
                    setSettingsAttributeMeta(response);
                    if (attributeLabel == '') {
                        setAttributeLabel(response.metaAttributeLabel);
                    }
                }
            });
    }, [queryParams]);

    const onchange = (input: InputType) => {
        dispatch(settingStore).setMetaFormData({
            ...settingsMetaFormData,
            attribute_id: settingsAttributeMeta.metaAttributeId,
            [input.name]: input.value,
        });
    };

    const onSubmit = () => {
        dispatch(settingStore)
            .saveAttributeMeta(settingsMetaFormData)
            .then((respons: any) => {
                if (respons) {
                    setQueryParams({ ...queryParams });
                }
            });
        setIsOpen(false);
    };

    const settingAttributeSearchHandler = lodash.debounce((text: string) => {
        setQueryParams({
            ...queryParams,
            search: text,
        });
    }, 500);

    const attributeTrashHandler = (select: any) => {
        setTrash(select.value.value);
        setQueryParams({
            ...queryParams,
            status: select.value.value ? 'trash' : 'active',
        });
    };

    const metaTrashHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure move to trash?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(settingStore)
                    .deleteSettingsMeta({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            setQueryParams({ ...queryParams });
                        }
                    });
            },
        });
    };
    const metaPermanentDeleteHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure to delete permanently?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(settingStore)
                    .deletePermanentlySettingsMeta({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            setQueryParams({ ...queryParams });
                        }
                    });
            },
        });
    };
    const metaRestoreHandler = (id: number) => {
        SwalWarning({
            title: __('Are you sure?', 'ai-job-listing'),
            text: __('Are you sure to restore the settings?', 'ai-job-listing'),
            showCancelButton: true,
            showConfirmButton: true,
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            showLoaderOnConfirm: true,
            timer: 50000,
            preConfirm: () => {
                dispatch(settingStore)
                    .restoreSettingsMeta({
                        ids: [id],
                    })
                    .then((respons: any) => {
                        if (respons.status === 200) {
                            setQueryParams({ ...queryParams });
                        }
                    });
            },
        });
    };

    const metaPaginationHandler = (pages: any) => {
        if (pages != null) {
            setQueryParams({
                ...queryParams,
                page: pages,
            });
        }
    };

    const jobSettingTableHeader = [
        {
            key: 'value',
            title: 'Name',
        },
        {
            key: 'action',
            title: 'Action',
            class: 'w-[80px]',
        },
    ];
    sortAlphabetically(settingsAttributeMeta?.data, 'value');

    return (
        <>
            <PageTitle text={attributeLabel} backButton />
            <FilterBar
                leftItem={
                    <>
                        <SearchInput
                            name="attribute_search"
                            placeholder="Search"
                            onChange={(text) =>
                                settingAttributeSearchHandler(text)
                            }
                        />
                    </>
                }
                rightItem={
                    <>
                        <Input
                            type="select"
                            name="attribute_trash"
                            value={trash}
                            select2Style={selectStylesSM}
                            options={[
                                { value: false, label: 'Active' },
                                { value: true, label: 'Trash' },
                            ]}
                            onChange={attributeTrashHandler}
                        />
                        <div>
                            {attributeLabel ? (
                                <Button
                                    btnStyle="secondary"
                                    icon="tio-add"
                                    text="Add New"
                                    onClick={
                                        (openModal = () => {
                                            setIsOpen(true);
                                            dispatch(
                                                settingStore
                                            ).setMetaFormData({
                                                ...settingsMetaDefaultData,
                                            });
                                        })
                                    }
                                />
                            ) : null}
                        </div>
                    </>
                }
            ></FilterBar>
            {loadingAttributeMeta ? (
                <LoadingBox block={settingsAttributeMeta?.data?.length} />
            ) : (
                <Table
                    header={jobSettingTableHeader}
                    body={settingsAttributeMeta.data}
                >
                    <div customKey="action">
                        {(meta: any) => (
                            <div className="flex items-start justify-end gap-2">
                                {(meta?.deleted_at === null ? (
                                        <Button
                                            text="Edit"
                                            btnStyle="action"
                                            outline
                                            onClick={
                                                (openModal = () => {
                                                    setIsOpen(true);
                                                    dispatch(
                                                        settingStore
                                                    ).setSettingAttributeSingleMeta(
                                                        meta.id
                                                    );
                                                })
                                            }
                                        />
                                    ) : (
                                        <Button
                                            text="Restore"
                                            btnStyle="action"
                                            outline
                                            onClick={() =>
                                                metaRestoreHandler(meta.id)
                                            }
                                        />
                                    ))}
                                {(meta?.deleted_at === null ? (
                                        <Dropdown>
                                            <Menu.Item>
                                                <Button
                                                    text="Trash"
                                                    btnStyle="transparentSecondary"
                                                    outline
                                                    onClick={() =>
                                                        metaTrashHandler(
                                                            meta.id
                                                        )
                                                    }
                                                />
                                            </Menu.Item>
                                        </Dropdown>
                                    ) : (
                                        <Dropdown>
                                            <Menu.Item>
                                                <Button
                                                    text="Delete Permanantly"
                                                    btnStyle="transparentSecondary"
                                                    outline
                                                    onClick={() =>
                                                        metaPermanentDeleteHandler(
                                                            meta.id
                                                        )
                                                    }
                                                />
                                            </Menu.Item>
                                        </Dropdown>
                                    ))}
                            </div>
                        )}
                    </div>
                </Table>
            )}
            <Pagination
                perPage={settings?.posts_per_page}
                total={settingAttributeMetaTotal}
                paginate={(pages: number) => {
                    metaPaginationHandler(pages);
                }}
            />
            {/* Create Edit Modal */}
            <Modal
                isOpen={isOpen}
                closeModal={closeModal}
                areaClass="w-[600px]"
            >
                <div className="flex justify-between border-b border-border-mid pb-5 mb-5">
                    <h3 className="mb-0">{attributeLabel}</h3>
                    <button
                        onClick={closeModal}
                        className="text-[20px] leading-[1] absolute top-[20px] right-[20px] text-border-dark"
                    >
                        <i className="tio-clear"></i>
                    </button>
                </div>
                <div className="mb-4">
                    <Input
                        label={`${attributeLabel} Name`}
                        type="text"
                        name="value"
                        value={settingsMetaFormData.value}
                        className="!border-border-mid"
                        onChange={onchange}
                    />
                </div>
                <div>
                    <Button
                        text="Save Changes"
                        btnStyle="secondary"
                        size="mid"
                        onClick={onSubmit}
                    />
                </div>
            </Modal>
        </>
    );
};

export default JobSettingsAttributes;
