/**
 * External dependencies
 */

import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useForm, SubmitHandler, Controller } from 'react-hook-form';
import { ErrorMessage } from '@hookform/error-message';
import Select from 'react-select';
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies.
 */
import joblistingstore from '../../../data/careers';
import { selectStyles } from '../../../components/form/StyleClass';
import { sortAlphabetically } from '../../../utils/Global';

const FormStepOne = ({ jobData }: any) => {
    const [formBlocks, setFormBlocks] = useState<any>({});
    const [profileImage, setProfileImage] = useState<any>({});
    const dispatch = useDispatch();
    const {
        register,
        handleSubmit,
        control,
        setValue,
        formState: { errors },
    } = useForm<any>();

    // Get Cv Submit form dynamic fields
    const cvFormFields: any = useSelect(
        (select) => select(joblistingstore).getCvPostFormFields(),
        []
    );
    // Get cv submit form data
    const getCvFormData: object = useSelect(
        (select) => select(joblistingstore).getCvPostForm(),
        []
    );

    useEffect(() => {
        const findBlock = cvFormFields.filter(
            (data: any) => data.block_id === 1
        );
        if (!findBlock[0]) return;
        setFormBlocks(findBlock[0]);
    }, [cvFormFields]);

    // Generate Form field by type

    const handleKeyPress = (event: any) => {
        const charCode = event.which ? event.which : event.keyCode;
        const charValue = String.fromCharCode(charCode);

        // Allow numbers (0-9), special characters like parentheses, dashes, spaces, and plus
        if (!/[0-9()+\-\s]/.test(charValue)) {
            event.preventDefault();
        }
    };

    const generateFieldByType = (fields: any) => {
        if (fields.type === 'text' && fields.is_show) {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <input
                        type={fields.type}
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        defaultValue={fields?.value}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                        })}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'email') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <input
                        type={fields.type}
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        defaultValue={fields?.value}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                            pattern: {
                                // Ensures email contains @, . after @, and characters after .
                                value: /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/,
                                message: 'Please enter a valid email',
                            },
                        })}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'tel') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <input
                        type={fields.type}
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        defaultValue={fields?.value}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                            pattern: {
                                value: /^[0-9()+\-\s]+$/,
                                message: 'Please enter a valid phone number',
                            },
                            minLength: {
                                value: 10,
                                message:
                                    'Phone number must be at least 10 digit',
                            },
                        })}
                        onKeyPress={handleKeyPress}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'select') {
            sortAlphabetically(fields.value, 'label');
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <Controller
                        name={fields.name}
                        control={control}
                        rules={{
                            required: {
                                value: fields.required,
                                message: 'Select field is required',
                            },
                        }}
                        render={({ field }) => (
                            <Select
                                {...field}
                                styles={selectStyles}
                                isSearchable={false}
                                options={fields.value}
                                placeholder="-"
                            />
                        )}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'file' && fields.name === 'profile_image') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <div className="avatar-attachement-placeholder">
                        <div className="placeholder">
                            {profileImage.length ||
                            getCvFormData.profile_image ? (
                                <img
                                    src={
                                        profileImage[0]
                                            ? URL.createObjectURL(
                                                  profileImage[0]
                                              )
                                            : URL.createObjectURL(
                                                  getCvFormData.profile_image
                                              )
                                    }
                                    alt="Profile"
                                />
                            ) : (
                                <i className="tio-user"></i>
                            )}
                        </div>
                        <div className="attachement-input-wrap">
                            <input
                                type={fields.type}
                                accept=".jpg, .png, .jpeg"
                                {...register(fields.name, {
                                    onChange: (e) => {
                                        setProfileImage(e.target.files);
                                    },
                                    required: {
                                        value:
                                            jobData?.is_required_cv_photo ===
                                                '1' &&
                                            !getCvFormData?.profile_image
                                                ? true
                                                : false,
                                        message: 'Profile Image is required',
                                    },
                                    validate: {
                                        acceptedFormats: (value) => {
                                            if (!value[0]) return true; // Skip validation if no file is selected
                                            return (
                                                [
                                                    'image/jpeg',
                                                    'image/jpg',
                                                    'image/png',
                                                ].includes(value[0].type) ||
                                                'Only PNG & JPEG files are allowed.'
                                            );
                                        },
                                        acceptedSize: (value) => {
                                            if (!value[0]) return true; // Skip validation if no file is selected
                                            const acceptedSize = 2097152; // Define accepted file Sizes
                                            return (
                                                acceptedSize > value[0].size ||
                                                'File size should be less than 2 MB'
                                            ); // Check if selected file type is in the accepted formats
                                        },
                                    },
                                })}
                            />
                            <div className="attachement-input-label">
                                <i className="tio-upload_on_cloud"></i>
                                Upload Picture
                            </div>
                        </div>
                    </div>
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                    {errors.upload_pictures?.type === 'acceptedFormats' && (
                        <span className="form-field-error">
                            Please upload a valid formate
                        </span>
                    )}
                </div>
            );
        }
    };
    // Form Submit Handler
    const onSubmit: SubmitHandler<any> = async (formData) => {
        dispatch(joblistingstore).setCvPostForm({
            ...getCvFormData,
            ...formData,
            profile_image:
                formData?.profile_image[0] ?? getCvFormData?.profile_image,
        });
        dispatch(joblistingstore).setCvFormStep(2);
        dispatch(joblistingstore).setCvFormStepCompleted(2);
    };

    useEffect(() => {
        // Set default form values when component mounts or when formData changes
        Object.entries(getCvFormData).forEach(([fieldName, value]) => {
            setValue(fieldName, value);
        });
    }, [setValue]);

    return (
        <>
            <form
                onSubmit={handleSubmit(onSubmit)}
                encType="multipart/form-data"
                className="job-apply-form"
            >
                <div className="job-apply-step-one">
                    {formBlocks?.sub_blocks?.map((block: any) => (
                        <div
                            key={block.sub_block_id}
                            className="job-form-blocks"
                        >
                            <h4 className="form-block-title">
                                {block.sub_block_name}
                            </h4>
                            <div className="job-form-row">
                                {block.field_options?.map((field: any) =>
                                    generateFieldByType(field)
                                )}
                            </div>
                        </div>
                    ))}
                    <div className="button-group button-group-end">
                        <div className="apply-btn">
                            <button type="submit">Next</button>
                        </div>
                    </div>
                </div>
            </form>
        </>
    );
};

export default FormStepOne;
