/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { useForm, SubmitHandler, Controller } from 'react-hook-form';
import { ErrorMessage } from '@hookform/error-message';
import Select from 'react-select';

/**
 * Internal dependencies.
 */
import { __ } from '@wordpress/i18n';
import joblistingstore from '../../../data/careers';
import { inputStyle, selectStyles } from '../../../components/form/StyleClass';
import { slugify, sortAlphabetically } from '../../../utils/Global';

const FormStepTwo = () => {
    const [formBlocks, setFormBlocks] = useState<any>({});
    const dispatch = useDispatch();
    const {
        register,
        handleSubmit,
        control,
        setValue,
        formState: { errors },
    } = useForm<any>();

    // Cv Submit form dynamic fields
    const cvFormFields: any = useSelect(
        (select) => select(joblistingstore).getCvPostFormFields(),
        []
    );
    // Store cv submit form data
    const getCvFormData: object = useSelect(
        (select) => select(joblistingstore).getCvPostForm(),
        []
    );

    useEffect(() => {
        const findBlock = cvFormFields.filter(
            (data: any) => data.block_id === 2
        );
        if (!findBlock[0]) return;
        setFormBlocks(findBlock[0]);
    }, [cvFormFields]);

    const [conditionalField, setConditionalField] = useState<object>({});

    // Generate Form field by type
    const generateFieldByType = (fields: any) => {
        if (fields.type === 'text' || fields.type === 'number') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <input
                        type={fields.type}
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                        })}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'select') {
            sortAlphabetically(fields.value, 'label');
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <Controller
                        name={fields.name}
                        control={control}
                        rules={{
                            required: {
                                value: fields.required,
                                message: 'Select field is required',
                            },
                        }}
                        render={({ field }) => (
                            <Select
                                {...field}
                                styles={selectStyles}
                                options={fields.value}
                                isSearchable={false}
                                placeholder="-"
                            />
                        )}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'textarea') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <textarea
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        rows={7}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                        })}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'radio') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>{`${fields.label} ${
                        fields.required ? '*' : ''
                    }`}</label>
                    <div className="checkbox-inline-wrap">
                        <label
                            htmlFor={`${fields.name}-${slugify(
                                fields.value.no
                            )}`}
                            className="checkbox-bordered"
                        >
                            <input
                                type={fields.type}
                                placeholder={fields.placeholder}
                                id={`${fields.name}-${slugify(
                                    fields.value.no
                                )}`}
                                value={slugify(fields.value.no)}
                                {...register(fields.name, {
                                    required: {
                                        value: fields.required,
                                        message: 'Field is required',
                                    },
                                })}
                            />
                            <label
                                htmlFor={`${fields.name}-${slugify(
                                    fields.value.no
                                )}`}
                            >{`${fields.value.no}`}</label>
                        </label>
                        <label
                            htmlFor={`${fields.name}-${slugify(
                                fields.value.yes
                            )}`}
                            className="checkbox-bordered flex items-center !border !border-solid !border-border-dark !rounded px-[20px] py-[8px] cursor-pointer"
                        >
                            <input
                                type={fields.type}
                                placeholder={fields.placeholder}
                                id={`${fields.name}-${slugify(
                                    fields.value.yes
                                )}`}
                                value={slugify(fields.value.yes)}
                                {...register(fields.name, {
                                    required: {
                                        value: fields.required,
                                        message: 'Field is required',
                                    },
                                    onChange: (e: any) => {
                                        setConditionalField({
                                            ...conditionalField,
                                            [fields.name]: e.target.value,
                                        });
                                    },
                                })}
                            />
                            <label
                                htmlFor={`${fields.name}-${slugify(
                                    fields.value.yes
                                )}`}
                            >{`${fields.value.yes}`}</label>
                        </label>
                    </div>
                    {fields.conditional_field?.type == 'textarea' &&
                        conditionalField[fields.name] == 'yes' && (
                            <div className="employee-list-textarea">
                                <textarea
                                    className={inputStyle}
                                    placeholder={
                                        fields.conditional_field?.placeholder
                                    }
                                    rows={4}
                                    {...register(
                                        fields.conditional_field?.name,
                                        {
                                            required: {
                                                value: true,
                                                message:
                                                    fields.conditional_field
                                                        ?.label +
                                                    ' is required',
                                            },
                                        }
                                    )}
                                />
                                <ErrorMessage
                                    errors={errors}
                                    name={fields.conditional_field?.name}
                                    render={({ message }) => (
                                        <span className="form-field-error">
                                            {message}
                                        </span>
                                    )}
                                />
                            </div>
                        )}
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        } else if (fields.type === 'url') {
            return (
                <div
                    key={fields.name}
                    className={
                        fields.half_width === false
                            ? 'job-form-col form-col-span-full'
                            : 'job-form-col'
                    }
                >
                    <label htmlFor={fields.name}>
                        {`${fields.label} ${fields.required ? '*' : ''}`}
                    </label>
                    <input
                        type="text"
                        className="job-form-input"
                        placeholder={fields.placeholder}
                        {...register(fields.name, {
                            required: {
                                value: fields.required,
                                message: fields.label + ' is required',
                            },
                            pattern: {
                                value: /^(https?:\/\/)?(www\.)?[^\s$.?#].[^\s]*\.[a-zA-Z]{2,}(\/[^\s]*)?$/,
                                message: 'Enter a valid URL',
                            },
                        })}
                    />
                    <ErrorMessage
                        errors={errors}
                        name={fields.name}
                        render={({ message }) => (
                            <span className="form-field-error">{message}</span>
                        )}
                    />
                </div>
            );
        }
    };

    // Form Submit Handler
    const onSubmit: SubmitHandler<any> = (formData) => {
        dispatch(joblistingstore).setCvPostForm({
            ...getCvFormData,
            ...formData,
        });
        dispatch(joblistingstore).setCvFormStep(3);
        dispatch(joblistingstore).setCvFormStepCompleted(3);
    };

    useEffect(() => {
        // Set default form values when component mounts or when formData changes
        Object.entries(getCvFormData).forEach(([fieldName, value]) => {
            setValue(fieldName, value);
        });
    }, [setValue]);

    return (
        <>
            <form
                onSubmit={handleSubmit(onSubmit)}
                encType="multipart/form-data"
                className="job-apply-form"
            >
                <div className="job-apply-step-two">
                    {formBlocks?.sub_blocks?.map((block: any) => (
                        <div
                            key={block.sub_block_id}
                            className="job-form-blocks"
                        >
                            <h4 className="form-block-title">
                                {block.sub_block_name}
                            </h4>
                            <div className="job-form-row">
                                {block.field_options?.map((field: any) =>
                                    generateFieldByType(field)
                                )}
                            </div>
                        </div>
                    ))}
                    <div className="button-group">
                        <div className="back-btn">
                            <button
                                type="button"
                                onClick={() =>
                                    dispatch(joblistingstore).setCvFormStep(1)
                                }
                            >
                                Back
                            </button>
                        </div>
                        <div className="apply-btn">
                            <button type="submit">Next</button>
                        </div>
                    </div>
                </div>
            </form>
        </>
    );
};

export default FormStepTwo;
