/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { Link, useParams } from 'react-router-dom';
/**
 * Internal dependencies.
 */
import careerStore from '../../../data/careers';
import { formatSalaryAmount, getValueByKey } from '../../../utils/Global';
import Loader from '../../../components/loader/Loader';
import { formatDateOnly } from '../../../utils/DateHelper';
import Error404 from '../Error404';

const JobDetails = ({ baseUrl, settings }: any) => {
    const dispatch = useDispatch();
    const params = useParams();
    const postSlug = params.slug;
    const [jobDetails, setJobDetails] = useState<any>({});
    const [pageError, setPageError] = useState<boolean>(false);

    const jobLoading: boolean = useSelect(
        (select) => select(careerStore).getJobsLoading(),
        []
    );

    useEffect(() => {
        dispatch(careerStore)
            .getJobs({
                slug: postSlug,
                status: ['active'],
            })
            .then((response: any) => {
                if (response?.[0]?.slug == postSlug) {
                    setJobDetails(response?.[0]);
                } else {
                    setPageError(true);
                }
            });
    }, [postSlug]);

    return (
        <>
            {jobLoading ? (
                <Loader />
            ) : (
                <>
                    {pageError ? (
                        <Error404 baseUrl={baseUrl} />
                    ) : (
                        <div className="ai-job-listing-details">
                            <div className="job-details-header">
                                <h1 className="job-title">
                                    {jobDetails?.title}
                                </h1>
                                <div className="job-meta-list">
                                    {getValueByKey(
                                        jobDetails?.attribute_values,
                                        'job_departments'
                                    ) && (
                                        <div className="job-meta">
                                            <i className="tio-briefcase_outlined"></i>
                                            Department:{' '}
                                            {getValueByKey(
                                                jobDetails?.attribute_values,
                                                'job_departments'
                                            )}
                                        </div>
                                    )}
                                    {getValueByKey(
                                        jobDetails?.attribute_values,
                                        'job_locations'
                                    ) && (
                                        <div className="job-meta">
                                            <i className="tio-poi_outlined"></i>
                                            Location:{' '}
                                            {getValueByKey(
                                                jobDetails?.attribute_values,
                                                'job_locations'
                                            )}
                                        </div>
                                    )}
                                </div>
                                {jobDetails?.slug && (
                                    <div className="social-share-links">
                                        <span className="share-label">Share:</span>
                                        <div className="social-links">
                                            <a
                                                href={`https://www.facebook.com/sharer/sharer.php?u=${window.location.href}`}
                                            >
                                                <i className="tio-facebook"></i>
                                            </a>
                                            <a
                                                href={`https://twitter.com/share?url=${window.location.href}`}
                                            >
                                                <i className="tio-twitter"></i>
                                            </a>
                                            <a
                                                href={`https://www.linkedin.com/shareArticle?mini=true&url=${window.location.href}`}
                                            >
                                                <i className="tio-linkedin"></i>
                                            </a>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.slug && (
                                    <>
                                        <div className="apply-btn">
                                            <Link
                                                to={`${baseUrl}/${jobDetails?.slug}/apply`}
                                            >
                                                APPLY NOW
                                            </Link>
                                        </div>
                                        <div className="back-list-btn">
                                            <Link to={`${baseUrl}`}>
                                                <i className="tio-chevron_left"></i>
                                                Back to Listing
                                            </Link>
                                        </div>
                                    </>
                                )}
                            </div>
                            <div className="job-details-content">
                                {jobDetails?.description && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Job Description</h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.description,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                {jobDetails?.responsibilities && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Job Responsibilities</h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.responsibilities,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                {jobDetails?.requirements && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Primary Requirements</h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.requirements,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                {jobDetails?.additional_requirements && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>
                                                Optional/Preferred Requirements
                                            </h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.additional_requirements,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                {jobDetails?.experienced_year && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Experience</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {
                                                    jobDetails?.experienced_year
                                                        ?.value
                                                }
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.attribute_values && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Job Location</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {getValueByKey(
                                                    jobDetails?.attribute_values,
                                                    'job_locations'
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.attribute_values && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Educational Requirements</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {getValueByKey(
                                                    jobDetails?.attribute_values,
                                                    'job_education_qualifications'
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.attribute_values && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Job Type</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {getValueByKey(
                                                    jobDetails?.attribute_values,
                                                    'job_types'
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.attribute_values && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Shift</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {getValueByKey(
                                                    jobDetails?.attribute_values,
                                                    'job_shifts'
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.attribute_values && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Department</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {getValueByKey(
                                                    jobDetails?.attribute_values,
                                                    'job_departments'
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.min_salary > 0 &&
                                    jobDetails.max_salary > 0 && (
                                        <div className="job-details-item">
                                            <div className="job-details-item-label">
                                                <h3>Salary Range</h3>
                                            </div>
                                            <div className="job-details-item-value">
                                                <p className="salary-range">
                                                    <span className="currency-symbol">
                                                        {
                                                            settings?.currency_symbol
                                                        }
                                                    </span>
                                                    <span>
                                                        {formatSalaryAmount(
                                                            jobDetails.min_salary
                                                        )}
                                                    </span>
                                                    <span className="seperator">
                                                        -
                                                    </span>
                                                    <span className="currency-symbol">
                                                        {
                                                            settings?.currency_symbol
                                                        }
                                                    </span>
                                                    <span>
                                                        {formatSalaryAmount(
                                                            jobDetails.max_salary
                                                        )}
                                                    </span>
                                                    <span className="salary-type">
                                                        {`(
                                                            ${getValueByKey(
                                                                jobDetails?.attribute_values,
                                                                'job_salary_type'
                                                            )}
                                                        )`}
                                                    </span>
                                                </p>
                                            </div>
                                        </div>
                                    )}
                                {jobDetails?.benefits && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Benefits</h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.benefits,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                {jobDetails?.is_required_cv_photo === '1' && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Photograph</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                Photograph must be enclosed with the Resume
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.application_deadline && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Application Deadline</h3>
                                        </div>
                                        <div className="job-details-item-value">
                                            <p>
                                                {formatDateOnly(
                                                    jobDetails?.application_deadline
                                                )}
                                            </p>
                                        </div>
                                    </div>
                                )}
                                {jobDetails?.additional_notes && (
                                    <div className="job-details-item">
                                        <div className="job-details-item-label">
                                            <h3>Additional Notes</h3>
                                        </div>
                                        <div
                                            className="job-details-item-value"
                                            dangerouslySetInnerHTML={{
                                                __html: jobDetails?.additional_notes,
                                            }}
                                        ></div>
                                    </div>
                                )}
                                <div className="job-details-item">
                                    <div className="job-details-item-label"></div>
                                    <div className="job-details-item-value">
                                        {jobDetails?.slug && (
                                            <div className="apply-btn">
                                                <Link
                                                    to={`${baseUrl}/${jobDetails?.slug}/apply`}
                                                >
                                                    APPLY NOW
                                                </Link>
                                            </div>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    )}
                </>
            )}
        </>
    );
};

export default JobDetails;
