/**
 * External dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { useSelect, useDispatch } from '@wordpress/data';
import { Link } from 'react-router-dom';
/**
 * Internal dependencies.
 */
import { selectStylesMd } from '../../../components/form/StyleClass';
import JobListingSidebar from './JobListingSidebar';
import { formatDateOnly } from '../../../utils/DateHelper';
import { displayedPageItemNumber, getValueByKey } from '../../../utils/Global';
import { Input } from '../../../components/form/Input';
import LoaderJobBox from '../../../components/loader/LoaderJobBox';
import Pagination from '../../../components/pagination/Pagination';
import careerStore from '../../../data/careers';

const JobListing = ({ baseUrl, settings }: any) => {
    const dispatch = useDispatch();
    const [jobSearchText, setJobSearchText] = useState<string>('');
    const [jobLists, setJobLists] = useState<Array<any>>([]);
    const [queryParams, setQueryParams] = useState<any>({
        column: '*',
        status: ['active'],
    });
    const [jobSortingSelect, setJobSortingSelect] = useState('most_recent');
    const [jobPerPageSelect, setJobPerPageSelect] = useState('10');
    const [jobPage, setJobPage] = useState<any>(1);
    const [pageItemsNumber, setPageItemsNumber] = useState<any>({});

    const jobTotalCount: any = useSelect(
        (select) => select(careerStore).getJobsTotalCount(),
        []
    );
    const jobListLoading: any = useSelect(
        (select) => select(careerStore).getJobsLoading(),
        []
    );

    const getParmasFromSidebar = (params: any) => {
        if (params.job_attributes !== null) {
            setQueryParams({
                ...queryParams,
                job_attributes: params.job_attributes,
            });
        }
    };

    useEffect(() => {
        dispatch(careerStore)
            .getJobs(queryParams)
            .then((response: any) => {
                if (response) {
                    setJobLists(response);
                }
            });
    }, [queryParams]);

    const jobSearchSubmitHandler = (e: any) => {
        e.preventDefault();
        if (jobSearchText.length >= 1) {
            setQueryParams({
                ...queryParams,
                search: jobSearchText,
                page: '1',
            });
        }
    };

    const jobSortHandler = (sort: any) => {
        if (jobSortingSelect !== sort.value.value) {
            setQueryParams({
                ...queryParams,
                sort_by: sort.value.value,
                page: '1',
            });
        }
        setJobSortingSelect(sort.value.value);
    };
    const jobPerPageHandler = (page: any) => {
        setJobPerPageSelect(page.value.value);
        if (jobPerPageSelect !== page.value.value) {
            setQueryParams({
                ...queryParams,
                page: '1',
                posts_per_page: page.value.value,
            });
        }
    };

    const jobPaginationHandler = (pages: any) => {
        setJobPage(pages || 1);
        if (pages != null) {
            setQueryParams({ ...queryParams, page: pages });
        }
    };

    useEffect(() => {
        const jobCount = displayedPageItemNumber(
            jobTotalCount,
            settings?.posts_per_page,
            jobPage
        );
        setPageItemsNumber(jobCount);
    }, [jobPage, jobTotalCount, settings]);

    // Filter sidbar control in Mobile
    const [windowWidth, setWindowWidth] = useState(window.outerWidth);
    const [filterToggle, setFilterToggle] = useState(false);

    addEventListener('resize', () => {
        setWindowWidth(window.outerWidth);
    });

    return (
        <>
            <div className="ai-job-listing-list">
                <div className="job-listing-header">
                    <h2 className="header-title">
                        {settings?.career_page_title}
                    </h2>
                    <p className="header-tagline">
                        {settings?.career_page_tagline}
                    </p>
                    <form
                        onSubmit={(e) => jobSearchSubmitHandler(e)}
                        className="job-search-form"
                    >
                        <div className="form-wrap">
                            <input
                                type="search"
                                name="job_search"
                                value={jobSearchText}
                                placeholder="Search job..."
                                onChange={(e) => {
                                    setJobSearchText(e.target.value);
                                    if (e.target.value.length < 1) {
                                        setQueryParams({
                                            ...queryParams,
                                            search: e.target.value,
                                        });
                                    }
                                }}
                            />
                            <button type="submit">Search</button>
                        </div>
                    </form>
                </div>
                <div className="job-listing-content">
                    <div
                        className={`job-listing-sidebar ${
                            filterToggle && windowWidth < 1024 ? 'active' : ''
                        }`}
                    >
                        <button
                            onClick={() => setFilterToggle(!filterToggle)}
                            className="sidebar-mobile-toggle"
                        >
                            <i className="tio-clear"></i> Close
                        </button>
                        <JobListingSidebar
                            setting={settings}
                            sendParamToParent={getParmasFromSidebar}
                        />
                    </div>
                    <div className="job-list-wrap">
                        <div className="job-list-header">
                            <div className="header-result-text">
                                <h3 className="result-found">
                                    {jobTotalCount}{' '}
                                    {jobTotalCount > 1 ? 'Jobs' : 'Job'} Found
                                </h3>
                                <button
                                    onClick={() =>
                                        setFilterToggle(!filterToggle)
                                    }
                                    className="sidebar-filter-toggle"
                                >
                                    <i className="tio-filter_outlined"></i>{' '}
                                    Filter
                                </button>
                            </div>
                            <div className="header-filters">
                                <div className="result-display">
                                    Displayed Here:{' '}
                                    {jobPerPageSelect > jobTotalCount
                                        ? pageItemsNumber?.from +
                                          '-' +
                                          jobTotalCount
                                        : pageItemsNumber?.from +
                                          '-' +
                                          pageItemsNumber.to}{' '}
                                    Jobs
                                </div>
                                <div className="filter-items">
                                    <Input
                                        type="select"
                                        name="sort_by"
                                        select2Style={selectStylesMd}
                                        value={jobSortingSelect}
                                        options={[
                                            {
                                                value: 'most_recent',
                                                label: 'Most Recent',
                                            },
                                            {
                                                value: 'alphabetically',
                                                label: 'Alphabetically',
                                            },
                                        ]}
                                        onChange={(sort) => {
                                            jobSortHandler(sort);
                                        }}
                                    />
                                    <Input
                                        type="select"
                                        name="job_per_page"
                                        select2Style={selectStylesMd}
                                        value={jobPerPageSelect}
                                        options={[
                                            {
                                                value: '10',
                                                label: '10 Per Page',
                                            },
                                            {
                                                value: '20',
                                                label: '20 Per Page',
                                            },
                                            {
                                                value: '50',
                                                label: '50 Per Page',
                                            },
                                        ]}
                                        onChange={(page) => {
                                            jobPerPageHandler(page);
                                        }}
                                    />
                                </div>
                            </div>
                        </div>
                        <div className="job-box-lists">
                            {jobListLoading ? (
                                <LoaderJobBox block={3} />
                            ) : (
                                jobLists?.map((job) => (
                                    <div key={job.id} className="job-box">
                                        <div className="job-box-inner">
                                            <h4 className="job-title">
                                                <Link
                                                    to={`${baseUrl}/${job.slug}/`}
                                                >
                                                    {job.title}
                                                </Link>
                                            </h4>
                                            <p
                                                className="job-excerpt"
                                                dangerouslySetInnerHTML={{
                                                    __html: job?.description,
                                                }}
                                            ></p>
                                            <div className="job-meta-list">
                                                {getValueByKey(
                                                    job.attribute_values,
                                                    'job_departments'
                                                ) && (
                                                    <div className="job-meta">
                                                        <i className="tio-briefcase_outlined"></i>
                                                        <span>
                                                            {getValueByKey(
                                                                job.attribute_values,
                                                                'job_departments'
                                                            )}
                                                        </span>
                                                    </div>
                                                )}
                                                {job?.application_deadline && (
                                                    <div className="job-meta">
                                                        <i className="tio-appointment"></i>
                                                        <span>
                                                            {formatDateOnly(
                                                                job?.application_deadline
                                                            )}
                                                        </span>
                                                    </div>
                                                )}
                                                {getValueByKey(
                                                    job.attribute_values,
                                                    'job_locations'
                                                ) && (
                                                    <div className="job-meta">
                                                        <i className="tio-poi_outlined"></i>
                                                        <span>
                                                            {getValueByKey(
                                                                job.attribute_values,
                                                                'job_locations'
                                                            )}
                                                        </span>
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                    </div>
                                ))
                            )}
                        </div>
                        <div>
                            <Pagination
                                perPage={
                                    settings?.posts_per_page >= jobPerPageSelect
                                        ? settings?.posts_per_page
                                        : jobPerPageSelect
                                }
                                total={jobTotalCount}
                                paginate={(pages: number) => {
                                    jobPaginationHandler(pages);
                                }}
                            />
                        </div>
                    </div>
                </div>
            </div>
            {filterToggle && windowWidth < 1024 ? (
                <button
                    onClick={() => setFilterToggle(!filterToggle)}
                    className="filter-toggle-mask"
                ></button>
            ) : (
                ''
            )}
        </>
    );
};

export default JobListing;
