import { format } from 'date-fns';

export const slugify = (text: any) => {
    return text
        .toString()
        .toLowerCase()
        .replace(/\s+/g, '-') // Replace spaces with -
        .replace(/[^\w-]+/g, '') // Remove all non-word chars
        .replace(/--+/g, '-') // Replace multiple - with single -
        .replace(/^-+/, '') // Trim - from start of text
        .replace(/-+$/, ''); // Trim - from end of text
};

export const unslugify = (slug: any) => {
    return slug
        .replace(/-/g, ' ') // Replace - with space
        .replace(/(?:^|\s)\S/g, function (a: any) {
            return a.toUpperCase();
        }); // Convert first letter of each word to uppercase
};

// Formate Salary Number
export const formatSalaryAmount = (number: any) => {
    // Check if the input is a valid number
    if (isNaN(number)) {
        // console.log('Invalid number input');
        return number;
    }
    // Convert the input to a number (in case it's a string or other type)
    const numericValue = Number(number);

    // Use toLocaleString to format the number
    return numericValue.toLocaleString('en-IN', {
        minimumFractionDigits: 0,
        maximumFractionDigits: 0,
    });
};

// Function to filter duplicate objects based on a ID
export const filterDuplicatesObject = (
    array: Array<any>,
    id: number | string
) => {
    return array.filter(
        (obj, index, self) => index === self.findIndex((t) => t[id] === obj[id])
    );
};

// Merge and filter duplicate value and key between 2 arrary If exist same key between arrays

export const mergeAndFilterDuplicateArraysByKey = (
    arr1: Array<any>,
    arr2: Array<any>,
    key: string
) => {
    // Create a map to store elements of arr2 by key
    const map = new Map(arr2.map((item) => [item[key], item]));

    // Map over arr1 and merge elements from arr2 if they have the same key
    return arr1.map((item) => ({
        ...item,
        ...(map.get(item[key]) || {}), // Merge with item from arr2 if found, otherwise merge with empty object
    }));
};

// Merge 2 Arrary
export const mergeTwoArray = (arrayOne: Array<any>, arrayTwo: Array<any>) => {
    // Convert array two into a map for easier lookup
    const mapTwo = new Map(arrayTwo.map((obj) => [obj.date, obj.application]));
    // console.log(mapTwo.get(obj.date))

    // Merge arrays
    const mergedArray = arrayOne.map((obj) => {
        const mergedObj = { ...obj };
        const matchingApplication = mapTwo.get(obj.date);
        mergedObj.application =
            matchingApplication !== undefined ? matchingApplication : 0;
        return mergedObj;
    });
    return mergedArray;
};

// Prepare Apply form data
export const prepareDataForApplyForm = (formData: any) => {
    let preparedData: any = [];

    Object.entries(formData).map(([key, value]: any) => {
        let metaValue = value;
        if (typeof value === 'object' && value?.value) {
            metaValue = value.value;
        } else if (value instanceof File) {
            metaValue = value;
        }
        const metas = {
            app_mk: key,
            app_mv: metaValue,
        };
        return preparedData.push(metas);
    });
    const filteredData = preparedData.filter(
        (item: any) => item.app_mv !== '' && item.app_mv !== null
    );
    preparedData = filteredData;

    return preparedData;
};

// Get Attribute valu by Key
export const getValueByKey = (array: string, key: string) => {
    if (array) {
        const stringToArray = JSON.parse(array);
        const formKey = key;
        if (stringToArray[0]) {
            for (let i = 0; i < stringToArray.length; i++) {
                if (stringToArray[i].form_key === formKey) {
                    return stringToArray[i].value;
                }
            }
        }
    }
};

// Get Applicant Data by key
export const getApplicantValueByKey = (array: string, key: string) => {
    if (array) {
        const stringToArray = JSON.parse(array);
        const formKey = key;
        if (stringToArray) {
            for (let i = 0; i < stringToArray.length; i++) {
                if (stringToArray[i].app_mk === formKey) {
                    return stringToArray[i].app_mv;
                }
            }
        }
    }
};

// Get formated Url
export const getFormattedUrl = (array: string, key: string) => {
    if (array) {
        const stringToArray = JSON.parse(array);
        const formKey = key;
        if (stringToArray) {
            for (let i = 0; i < stringToArray.length; i++) {
                if (stringToArray[i].app_mk === formKey) {
                    let url = stringToArray[i].app_mv;
                    if (!url.startsWith('https')) {
                        url = 'https://' + url;
                    }
                    return url;
                }
            }
        }
    }
};


// Get Attribute id by Key
export const getIdByKey = (array: string, key: string) => {
    if (array) {
        const stringToArray = JSON.parse(array);
        const formKey = key;
        if (stringToArray[0]) {
            for (let i = 0; i < stringToArray.length; i++) {
                if (stringToArray[i].form_key === formKey) {
                    return stringToArray[i].id;
                }
            }
        }
    }
};

// Get Application Details by Key
export const getApplicationDetailsWithStatusById = (
    payload: any,
    id: string
) => {
    try {
        const appDetails = {
            application: {},
            nextStatus: payload.nextStatus,
            prevStatus: payload.prevStatus,
        };
        const appData = payload.applications.find((item: any) => item.id == id);
        if (!appData) {
            throw new Error('No Data Found!');
        } else {
            appDetails.application = appData;
        }
        return appDetails;
    } catch (error) {
        // eslint-disable-next-line no-console
        console.log('Error in getting application details', error);
    }
};
// Get Application Details by Key
export const filterArrayItemById = (payload: any, id: any) => {
    try {
        let filteredItem = [];
        const appData = payload.filter((item: any) => item.id != id);
        if (!appData) {
            throw new Error('No Data Found!');
        } else {
            filteredItem = appData;
        }
        return filteredItem;
    } catch (error) {
        // eslint-disable-next-line no-console
        console.log('Error in getting filtered item', error);
    }
};

export const getApplicationCountByDate = (applications: Array<any>) => {
    const itemCount: any = {};
    applications.filter((item) => {
        const { date } = item;
        if (!itemCount[date]) {
            itemCount[date] = 1;
            return true; // Keep the first occurrence
            // eslint-disable-next-line no-else-return
        } else {
            itemCount[date]++;
            return false; // Remove duplicate items
        }
    });
    const dateCountArray = Object.entries(itemCount).map(
        ([date, application]) => {
            const formateDate = new Date(date);
            return { date: format(formateDate, 'MMM dd, yyyy'), application };
        }
    );
    return dateCountArray;
};

// Trim Last word
export const trimLastWord = (str: string) => {
    if (str) {
        // Split the string into an array of words
        const words = str.split('');
        // Remove the last element from the array
        words.pop();
        // Join the remaining elements back into a string
        return words.join('');
    }
};

export const displayedPageItemNumber = (
    total = 0,
    perPage = 0,
    pageNumber = 0
) => {
    // Calculate the index of the first item on the page
    const from = (pageNumber - 1) * perPage + 1;

    // Calculate the index of the last item on the page
    let to = pageNumber * perPage;

    // Ensure the "to" index does not exceed the total number of items
    if (to > total) {
        to = total;
    }

    // If the "from" index exceeds the total number of items, adjust it
    if (from > total) {
        return { from: 0, to: 0 };
    }

    return { from, to };
};

// Copy text Function
export const fallbackCopyTextHandler = async (text: any) => {
    // Create a temporary text area element to hold the text
    const tempTextArea = document.createElement('textarea');
    tempTextArea.value = text;

    // Set styles to ensure the text area isn't visible
    tempTextArea.style.position = 'fixed';
    tempTextArea.style.opacity = '0';
    tempTextArea.style.left = '-9999px';

    // Append the text area to the document body
    document.body.appendChild(tempTextArea);

    // Select the text in the text area
    tempTextArea.select();

    // Copy the text to the clipboard using execCommand('copy')
    try {
        document.execCommand('copy');
        // eslint-disable-next-line no-alert
        return {
            message: true,
        };
    } catch (err) {
        // eslint-disable-next-line no-alert
        console.error('Oops! Somethings went wrong for copied');
    }
    // Remove the temporary text area from the document
    document.body.removeChild(tempTextArea);
};

export const currenciesSymbol = [
    { value: '$', label: 'USD ($)' }, // United States Dollar
    { value: '€', label: 'EUR (€)' }, // Euro
    { value: '¥', label: 'JPY (¥)' }, // Japanese Yen
    { value: '£', label: 'GBP (£)' }, // British Pound Sterling
    { value: 'A$', label: 'AUD (A$)' }, // Australian Dollar
    { value: 'C$', label: 'CAD (C$)' }, // Canadian Dollar
    { value: 'CHF', label: 'CHF (CHF)' }, // Swiss Franc
    { value: '¥CNY', label: 'CNY (¥)' }, // Chinese Yuan
    { value: 'krSEK', label: 'SEK (kr)' }, // Swedish Krona
    { value: '$NZD', label: 'NZD ($)' }, // New Zealand Dollar
    { value: '$MXN', label: 'MXN ($)' }, // Mexican Peso
    { value: '$SGD', label: 'SGD ($)' }, // Singapore Dollar
    { value: '$HKD', label: 'HKD ($)' }, // Hong Kong Dollar
    { value: 'krNOK', label: 'NOK (kr)' }, // Norwegian Krone
    { value: '₩', label: 'KRW (₩)' }, // South Korean Won
    { value: '₺', label: 'TRY (₺)' }, // Turkish Lira
    { value: '₽', label: 'RUB (₽)' }, // Russian Ruble
    { value: '₹', label: 'INR (₹)' }, // Indian Rupee
    { value: 'R$', label: 'BRL (R$)' }, // Brazilian Real
    { value: 'R', label: 'ZAR (R)' }, // South African Rand
    { value: '₱', label: 'PHP (₱)' }, // Philippine Peso
    { value: 'zł', label: 'PLN (zł)' }, // Polish Zloty
    { value: '฿', label: 'THB (฿)' }, // Thai Baht
    { value: 'Rp', label: 'IDR (Rp)' }, // Indonesian Rupiah
    { value: 'Kč', label: 'CZK (Kč)' }, // Czech Koruna
    { value: '₪', label: 'ILS (₪)' }, // Israeli New Shekel
    { value: 'RM', label: 'MYR (RM)' }, // Malaysian Ringgit
    { value: 'Ft', label: 'HUF (Ft)' }, // Hungarian Forint
    { value: 'د.إ', label: 'AED (د.إ)' }, // UAE Dirham
    { value: 'ر.س', label: 'SAR (ر.س)' }, // Saudi Riyal
    { value: '৳', label: 'BDT (৳)' }, // Bangladeshi Taka
    { value: '₨PKR', label: 'PKR (₨)' }, // Pakistani Rupee
    { value: '₨LKR', label: 'LKR (₨)' }, // Sri Lankan Rupee
    { value: '₦', label: 'NGN (₦)' }, // Nigerian Naira
    { value: 'E£', label: 'EGP (E£)' }, // Egyptian Pound
    { value: 'KSh', label: 'KES (KSh)' }, // Kenyan Shilling
    { value: '₫', label: 'VND (₫)' }, // Vietnamese Dong
    { value: '$COP', label: 'COP ($)' }, // Colombian Peso
    { value: '$ARS', label: 'ARS ($)' }, // Argentine Peso
    { value: '$CLP', label: 'CLP ($)' }, // Chilean Peso
    { value: 'S/', label: 'PEN (S/)' }, // Peruvian Sol
    { value: '₴', label: 'UAH (₴)' }, // Ukrainian Hryvnia
    { value: 'lei', label: 'RON (lei)' }, // Romanian Leu
    { value: 'دج', label: 'DZD (دج)' }, // Algerian Dinar
    { value: 'лв', label: 'KZT (лв)' }, // Kazakhstani Tenge
    { value: '﷼QAR', label: 'QAR (﷼)' }, // Qatari Rial
    { value: '.د.ب', label: 'BHD (.د.ب)' }, // Bahraini Dinar
    { value: '﷼OMR', label: 'OMR (﷼)' }, // Omani Rial
    { value: 'د.ا', label: 'JOD (د.ا)' }, // Jordanian Dinar
    { value: 'د.ك', label: 'KWD (د.ك)' }, // Kuwaiti Dinar
];

export const getArrayValueByKey = (array: Array<any>, name: string) => {
    const item = array.find((obj) => obj.name === name);
    return item ? item.value : null;
};
export const sortAlphabetically = (array: Array<any>, name: string) => {
    const sortedItem = array?.sort((a: any, b: any) =>
        a[name].localeCompare(b[name])
    );
    return sortedItem ? sortedItem : null;
};
